// MarkaJiap Lightbox - Full-screen image viewer
// Features: zoom, keyboard navigation, gallery mode

(function() {
  'use strict';
  
  console.log('📷 MarkaJiap Lightbox: Loaded');
  
  class MarkaJiapLightbox {
    constructor() {
      this.images = [];
      this.currentIndex = 0;
      this.zoomLevel = 1;
      this.isOpen = false;
      this.overlay = null;
      
      this.init();
    }
    
    init() {
      this.createOverlay();
      this.bindEvents();
      console.log('📷 MarkaJiap Lightbox: Ready');
    }
    
    createOverlay() {
      this.overlay = document.createElement('div');
      this.overlay.id = 'markajiap-lightbox';
      this.overlay.innerHTML = `
        <div class="lightbox-backdrop"></div>
        <div class="lightbox-container">
          <button class="lightbox-close" title="Close (Esc)">✕</button>
          <button class="lightbox-prev" title="Previous (←)">‹</button>
          <button class="lightbox-next" title="Next (→)">›</button>
          <div class="lightbox-content">
            <img class="lightbox-image" alt="">
          </div>
          <div class="lightbox-footer">
            <span class="lightbox-counter">1 / 1</span>
            <div class="lightbox-zoom">
              <button class="lightbox-zoom-out" title="Zoom out (-)">−</button>
              <span class="lightbox-zoom-level">100%</span>
              <button class="lightbox-zoom-in" title="Zoom in (+)">+</button>
              <button class="lightbox-zoom-reset" title="Reset (0)">⟲</button>
            </div>
            <span class="lightbox-caption"></span>
          </div>
        </div>
      `;
      
      this.injectStyles();
      document.body.appendChild(this.overlay);
    }
    
    injectStyles() {
      if (document.getElementById('markajiap-lightbox-styles')) return;
      
      const style = document.createElement('style');
      style.id = 'markajiap-lightbox-styles';
      style.textContent = `
        #markajiap-lightbox {
          display: none;
          position: fixed;
          top: 0;
          left: 0;
          width: 100vw;
          height: 100vh;
          z-index: 10000;
        }
        #markajiap-lightbox.open { display: flex; }
        
        .lightbox-backdrop {
          position: absolute;
          inset: 0;
          background: rgba(0, 0, 0, 0.9);
          backdrop-filter: blur(10px);
        }
        
        .lightbox-container {
          position: relative;
          display: flex;
          flex-direction: column;
          width: 100%;
          height: 100%;
        }
        
        .lightbox-content {
          flex: 1;
          display: flex;
          align-items: center;
          justify-content: center;
          overflow: auto;
          padding: 60px 80px;
        }
        
        .lightbox-image {
          max-width: 100%;
          max-height: 100%;
          object-fit: contain;
          transition: transform 0.2s ease;
          cursor: zoom-in;
          border-radius: 4px;
          box-shadow: 0 10px 50px rgba(0, 0, 0, 0.5);
        }
        .lightbox-image.zoomed { cursor: grab; }
        .lightbox-image.dragging { cursor: grabbing; }
        
        .lightbox-close,
        .lightbox-prev,
        .lightbox-next {
          position: absolute;
          background: rgba(255, 255, 255, 0.1);
          border: none;
          color: white;
          cursor: pointer;
          z-index: 1;
          transition: all 0.2s ease;
        }
        .lightbox-close:hover,
        .lightbox-prev:hover,
        .lightbox-next:hover {
          background: rgba(123, 45, 142, 0.8);
        }
        
        .lightbox-close {
          top: 16px;
          right: 16px;
          width: 40px;
          height: 40px;
          font-size: 24px;
          border-radius: 50%;
        }
        
        .lightbox-prev,
        .lightbox-next {
          top: 50%;
          transform: translateY(-50%);
          width: 48px;
          height: 80px;
          font-size: 36px;
          border-radius: 4px;
        }
        .lightbox-prev { left: 16px; }
        .lightbox-next { right: 16px; }
        .lightbox-prev:disabled,
        .lightbox-next:disabled {
          opacity: 0.3;
          cursor: not-allowed;
        }
        
        .lightbox-footer {
          display: flex;
          align-items: center;
          justify-content: space-between;
          padding: 12px 24px;
          background: rgba(0, 0, 0, 0.5);
          color: white;
          font-size: 14px;
        }
        
        .lightbox-zoom {
          display: flex;
          align-items: center;
          gap: 8px;
        }
        .lightbox-zoom button {
          width: 32px;
          height: 32px;
          background: rgba(255, 255, 255, 0.1);
          border: none;
          color: white;
          cursor: pointer;
          border-radius: 4px;
          font-size: 16px;
        }
        .lightbox-zoom button:hover {
          background: rgba(123, 45, 142, 0.8);
        }
        .lightbox-zoom-level {
          min-width: 50px;
          text-align: center;
        }
        
        .lightbox-caption {
          max-width: 40%;
          overflow: hidden;
          text-overflow: ellipsis;
          white-space: nowrap;
          opacity: 0.8;
        }
        
        .lightbox-counter {
          min-width: 60px;
        }
      `;
      document.head.appendChild(style);
    }
    
    bindEvents() {
      // Close button
      this.overlay.querySelector('.lightbox-close').addEventListener('click', () => this.close());
      
      // Backdrop click
      this.overlay.querySelector('.lightbox-backdrop').addEventListener('click', () => this.close());
      
      // Navigation
      this.overlay.querySelector('.lightbox-prev').addEventListener('click', () => this.prev());
      this.overlay.querySelector('.lightbox-next').addEventListener('click', () => this.next());
      
      // Zoom controls
      this.overlay.querySelector('.lightbox-zoom-in').addEventListener('click', () => this.zoom(0.25));
      this.overlay.querySelector('.lightbox-zoom-out').addEventListener('click', () => this.zoom(-0.25));
      this.overlay.querySelector('.lightbox-zoom-reset').addEventListener('click', () => this.resetZoom());
      
      // Image click to zoom
      this.overlay.querySelector('.lightbox-image').addEventListener('click', (e) => {
        if (this.zoomLevel === 1) {
          this.zoom(1); // Zoom to 200%
        } else {
          this.resetZoom();
        }
      });
      
      // Keyboard
      document.addEventListener('keydown', (e) => {
        if (!this.isOpen) return;
        
        switch (e.key) {
          case 'Escape': this.close(); break;
          case 'ArrowLeft': this.prev(); break;
          case 'ArrowRight': this.next(); break;
          case '+': case '=': this.zoom(0.25); break;
          case '-': this.zoom(-0.25); break;
          case '0': this.resetZoom(); break;
        }
      });
    }
    
    open(images, startIndex = 0) {
      this.images = images;
      this.currentIndex = startIndex;
      this.isOpen = true;
      this.resetZoom();
      
      this.overlay.classList.add('open');
      document.body.style.overflow = 'hidden';
      
      this.updateDisplay();
      console.log('📷 MarkaJiap Lightbox: Opened with', images.length, 'images');
    }
    
    close() {
      this.isOpen = false;
      this.overlay.classList.remove('open');
      document.body.style.overflow = '';
      console.log('📷 MarkaJiap Lightbox: Closed');
    }
    
    prev() {
      if (this.currentIndex > 0) {
        this.currentIndex--;
        this.resetZoom();
        this.updateDisplay();
      }
    }
    
    next() {
      if (this.currentIndex < this.images.length - 1) {
        this.currentIndex++;
        this.resetZoom();
        this.updateDisplay();
      }
    }
    
    zoom(delta) {
      this.zoomLevel = Math.max(0.5, Math.min(4, this.zoomLevel + delta));
      this.updateZoom();
    }
    
    resetZoom() {
      this.zoomLevel = 1;
      this.updateZoom();
    }
    
    updateZoom() {
      const img = this.overlay.querySelector('.lightbox-image');
      img.style.transform = `scale(${this.zoomLevel})`;
      img.classList.toggle('zoomed', this.zoomLevel > 1);
      
      this.overlay.querySelector('.lightbox-zoom-level').textContent = Math.round(this.zoomLevel * 100) + '%';
    }
    
    updateDisplay() {
      const img = this.overlay.querySelector('.lightbox-image');
      const current = this.images[this.currentIndex];
      
      img.src = current.src;
      img.alt = current.alt || '';
      
      // Update counter
      this.overlay.querySelector('.lightbox-counter').textContent = 
        `${this.currentIndex + 1} / ${this.images.length}`;
      
      // Update caption
      this.overlay.querySelector('.lightbox-caption').textContent = current.alt || current.src.split('/').pop();
      
      // Update nav buttons
      this.overlay.querySelector('.lightbox-prev').disabled = this.currentIndex === 0;
      this.overlay.querySelector('.lightbox-next').disabled = this.currentIndex === this.images.length - 1;
    }
  }
  
  // Initialize and expose globally
  window.MarkaJiapLightbox = new MarkaJiapLightbox();
})();
