// MarkaJiap Exporter - Export rendered markdown to HTML and DOCX
// This runs in page context

(function() {
  'use strict';
  
  console.log('📄 MarkaJiap Exporter: Loaded');
  
  class MarkaJiapExporter {
    constructor() {
      console.log('📄 MarkaJiap Exporter: Ready');
    }
    
    // Export current content to standalone HTML file
    exportToHTML(options = {}) {
      const content = document.getElementById('markajiap-content');
      if (!content) {
        console.error('📄 MarkaJiap Exporter: No content found');
        return;
      }
      
      const title = document.title || 'MarkaJiap Export';
      const theme = document.documentElement.getAttribute('data-theme') || 'light';
      
      // Get all styles
      const styles = this.collectStyles();
      
      // Build standalone HTML
      const html = `<!DOCTYPE html>
<html lang="en" data-theme="${theme}">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>${this.escapeHtml(title)}</title>
  <style>
${styles}

/* Additional export styles */
body {
  max-width: 900px;
  margin: 0 auto;
  padding: 40px 20px;
}
.markdown-body {
  padding: 0;
}
/* Hide interactive elements */
.code-copy-btn,
.mermaid-controls,
#markajiap-sidebar,
.toolbar,
.progress-bar,
#scroll-top {
  display: none !important;
}
  </style>
</head>
<body>
  <article class="markdown-body">
${content.innerHTML}
  </article>
  
  <footer style="margin-top: 60px; padding-top: 20px; border-top: 1px solid #EDE9F0; text-align: center; color: #6B6B7B; font-size: 12px;">
    Exported with 🔮 MarkaJiap
  </footer>
</body>
</html>`;

      // Trigger download
      this.downloadFile(html, this.getFilename() + '.html', 'text/html');
      console.log('📄 MarkaJiap Exporter: HTML exported');
    }
    
    // Export to DOCX (requires docx library)
    async exportToDOCX() {
      const content = document.getElementById('markajiap-content');
      if (!content) {
        console.error('📄 MarkaJiap Exporter: No content found');
        return;
      }
      
      // For now, use HTML-to-DOCX approach via Blob
      // Can be enhanced with proper docx.js library later
      const html = content.innerHTML;
      const title = document.title || 'MarkaJiap Export';
      
      // Create Word-compatible HTML
      const docContent = `
<!DOCTYPE html>
<html xmlns:o="urn:schemas-microsoft-com:office:office" 
      xmlns:w="urn:schemas-microsoft-com:office:word"
      xmlns="http://www.w3.org/TR/REC-html40">
<head>
  <meta charset="UTF-8">
  <title>${this.escapeHtml(title)}</title>
  <!--[if gte mso 9]>
  <xml>
    <w:WordDocument>
      <w:View>Print</w:View>
      <w:Zoom>100</w:Zoom>
    </w:WordDocument>
  </xml>
  <![endif]-->
  <style>
    body { font-family: Calibri, sans-serif; font-size: 11pt; line-height: 1.6; }
    h1 { font-size: 24pt; color: #7B2D8E; }
    h2 { font-size: 18pt; color: #5E2270; }
    h3 { font-size: 14pt; color: #7B2D8E; }
    pre { background: #f5f5f5; padding: 10pt; font-family: Consolas, monospace; font-size: 10pt; }
    code { font-family: Consolas, monospace; background: #f0f0f0; padding: 2pt 4pt; }
    blockquote { border-left: 4px solid #7B2D8E; padding-left: 10pt; color: #666; }
    table { border-collapse: collapse; width: 100%; }
    th, td { border: 1px solid #ddd; padding: 8pt; }
    th { background: #7B2D8E; color: white; }
    img { max-width: 100%; }
  </style>
</head>
<body>
${html}
</body>
</html>`;

      this.downloadFile(docContent, this.getFilename() + '.doc', 'application/msword');
      console.log('📄 MarkaJiap Exporter: DOCX exported');
    }
    
    // Collect all MarkaJiap styles
    collectStyles() {
      let styles = '';
      
      // Get markajiap-styles
      const mainStyle = document.getElementById('markajiap-styles');
      if (mainStyle) {
        styles += mainStyle.textContent + '\n';
      }
      
      // Get highlight styles
      const hljsStyles = document.querySelectorAll('style[id*="hljs"]');
      hljsStyles.forEach(s => {
        styles += s.textContent + '\n';
      });
      
      // Add essential variables (in case they're missing)
      const theme = document.documentElement.getAttribute('data-theme') || 'light';
      const isDark = theme === 'dark';
      
      styles += `
:root {
  --bg-primary: ${isDark ? '#16121F' : '#FFFFFF'};
  --bg-secondary: ${isDark ? '#1F1A2E' : '#FAF8FC'};
  --text-primary: ${isDark ? '#F0EDF5' : '#1A1625'};
  --text-secondary: ${isDark ? '#A8A8B8' : '#6B6B7B'};
  --accent: #7B2D8E;
  --accent-gold: #D4A634;
  --border-color: ${isDark ? '#2A243A' : '#EDE9F0'};
  --code-bg: ${isDark ? '#1F1A2E' : '#F6F3F8'};
}
[data-theme="dark"] {
  --bg-primary: #16121F;
  --bg-secondary: #1F1A2E;
  --text-primary: #F0EDF5;
  --text-secondary: #A8A8B8;
  --border-color: #2A243A;
  --code-bg: #1F1A2E;
}
`;
      
      return styles;
    }
    
    // Get filename from page title or URL
    getFilename() {
      let name = document.title || 'export';
      
      // Clean up
      name = name.replace(/[^a-zA-Z0-9\u0E00-\u0E7F\s-]/g, ''); // Allow Thai chars
      name = name.replace(/\s+/g, '_');
      name = name.substring(0, 50);
      
      return name || 'markajiap_export';
    }
    
    // Download file helper
    downloadFile(content, filename, mimeType) {
      const blob = new Blob([content], { type: mimeType + ';charset=utf-8' });
      const url = URL.createObjectURL(blob);
      
      const link = document.createElement('a');
      link.href = url;
      link.download = filename;
      link.click();
      
      setTimeout(() => URL.revokeObjectURL(url), 1000);
    }
    
    // Escape HTML for safe insertion
    escapeHtml(str) {
      const div = document.createElement('div');
      div.textContent = str;
      return div.innerHTML;
    }
    
    // Show export dialog
    showExportDialog() {
      // Remove existing dialog
      const existing = document.getElementById('markajiap-export-dialog');
      if (existing) existing.remove();
      
      const dialog = document.createElement('div');
      dialog.id = 'markajiap-export-dialog';
      dialog.innerHTML = `
        <div class="export-backdrop"></div>
        <div class="export-panel">
          <h3>📄 Export Document</h3>
          <div class="export-options">
            <button class="export-btn" data-format="html">
              <span class="export-icon">🌐</span>
              <span class="export-label">HTML</span>
              <span class="export-desc">Standalone web page</span>
            </button>
            <button class="export-btn" data-format="docx">
              <span class="export-icon">📝</span>
              <span class="export-label">Word</span>
              <span class="export-desc">Microsoft Word (.doc)</span>
            </button>
          </div>
          <button class="export-close">Cancel</button>
        </div>
      `;
      
      // Add styles
      const style = document.createElement('style');
      style.textContent = `
        #markajiap-export-dialog {
          position: fixed;
          inset: 0;
          display: flex;
          align-items: center;
          justify-content: center;
          z-index: 10001;
        }
        .export-backdrop {
          position: absolute;
          inset: 0;
          background: rgba(0,0,0,0.5);
          backdrop-filter: blur(4px);
        }
        .export-panel {
          position: relative;
          background: var(--bg-primary, white);
          border-radius: 12px;
          padding: 24px;
          min-width: 320px;
          box-shadow: 0 20px 60px rgba(0,0,0,0.3);
        }
        .export-panel h3 {
          margin: 0 0 20px;
          color: var(--text-primary);
        }
        .export-options {
          display: flex;
          gap: 12px;
          margin-bottom: 16px;
        }
        .export-btn {
          flex: 1;
          display: flex;
          flex-direction: column;
          align-items: center;
          gap: 8px;
          padding: 20px;
          background: var(--bg-secondary, #f5f5f5);
          border: 2px solid var(--border-color, #eee);
          border-radius: 8px;
          cursor: pointer;
          transition: all 0.2s;
        }
        .export-btn:hover {
          border-color: var(--accent, #7B2D8E);
          background: var(--bg-tertiary, #eee);
        }
        .export-icon { font-size: 32px; }
        .export-label { font-weight: 600; color: var(--text-primary); }
        .export-desc { font-size: 12px; color: var(--text-secondary); }
        .export-close {
          width: 100%;
          padding: 10px;
          background: transparent;
          border: 1px solid var(--border-color, #ddd);
          border-radius: 6px;
          color: var(--text-secondary);
          cursor: pointer;
        }
        .export-close:hover {
          background: var(--bg-secondary, #f5f5f5);
        }
      `;
      dialog.appendChild(style);
      document.body.appendChild(dialog);
      
      // Event handlers
      dialog.querySelector('.export-backdrop').addEventListener('click', () => dialog.remove());
      dialog.querySelector('.export-close').addEventListener('click', () => dialog.remove());
      
      dialog.querySelector('[data-format="html"]').addEventListener('click', () => {
        this.exportToHTML();
        dialog.remove();
      });
      
      dialog.querySelector('[data-format="docx"]').addEventListener('click', () => {
        this.exportToDOCX();
        dialog.remove();
      });
    }
  }
  
  // Initialize and expose globally
  window.MarkaJiapExporter = new MarkaJiapExporter();
})();
