// JVC MarkView - Viewer Script

class MarkViewViewer {
  constructor() {
    this.markdown = '';
    this.options = {};
    this.searchResults = [];
    this.currentSearchIndex = 0;
    this.init();
  }

  async init() {
    await this.loadOptions();
    await this.loadContent();
    this.setupUI();
    this.setupKeyboardShortcuts();
  }

  // Load user options
  async loadOptions() {
    return new Promise((resolve) => {
      chrome.storage.sync.get(null, (result) => {
        this.options = {
          theme: result.theme || 'light',
          showToc: result.showToc !== false,
          codeLineNumbers: result.codeLineNumbers !== false,
          enableMermaid: result.enableMermaid !== false,
          enableMath: result.enableMath !== false,
          fontSize: result.fontSize || 16,
          fontFamily: result.fontFamily || 'IBM Plex Sans',
          customCss: result.customCss || ''
        };
        
        // Apply theme
        document.documentElement.setAttribute('data-theme', this.options.theme);
        this.updateHighlightTheme();
        
        // Inject highlight.js syntax colors inline (guaranteed to work)
        this.injectHighlightStyles();
        
        // Apply font size
        document.documentElement.style.fontSize = this.options.fontSize + 'px';
        
        // Apply custom CSS
        if (this.options.customCss) {
          const style = document.createElement('style');
          style.textContent = this.options.customCss;
          document.head.appendChild(style);
        }
        
        resolve();
      });
    });
  }

  // Inject highlight.js syntax highlighting styles inline
  injectHighlightStyles() {
    const isDark = this.options.theme === 'dark';
    
    // Remove existing highlight styles
    const existingStyle = document.getElementById('hljs-inline-styles');
    if (existingStyle) existingStyle.remove();
    
    const style = document.createElement('style');
    style.id = 'hljs-inline-styles';
    
    if (isDark) {
      // Dark theme - MarkaJiap Lucky Colors
      style.textContent = `
        .hljs { color: #F0EDF5 !important; background: transparent !important; }
        .hljs-keyword, .hljs-type, .hljs-built_in, .hljs-doctag, .hljs-variable.language_ { color: #C77DFF !important; font-weight: 600; }
        .hljs-title, .hljs-title.class_, .hljs-title.function_ { color: #E0AAFF !important; font-weight: 600; }
        .hljs-string, .hljs-meta .hljs-string, .hljs-regexp { color: #E8C864 !important; }
        .hljs-number, .hljs-literal { color: #79C0FF !important; }
        .hljs-attr { color: #FF92DF !important; }
        .hljs-attribute, .hljs-variable { color: #58D1F0 !important; }
        .hljs-symbol, .hljs-bullet { color: #FFD700 !important; }
        .hljs-comment, .hljs-code, .hljs-formula { color: #8B949E !important; font-style: italic; }
        .hljs-name, .hljs-selector-tag, .hljs-quote { color: #7EE787 !important; }
        .hljs-operator, .hljs-meta { color: #FF79C6 !important; }
        .hljs-property { color: #79C0FF !important; }
        .hljs-params { color: #FFA657 !important; }
        .hljs-punctuation { color: #A09AAD !important; }
        .hljs-section { color: #C77DFF !important; font-weight: 700; }
        .hljs-subst { color: #F0EDF5 !important; }
        .hljs-addition { color: #AFF5B4 !important; background-color: #033A16 !important; }
        .hljs-deletion { color: #FFDCD7 !important; background-color: #67060C !important; }
      `;
    } else {
      // Light theme - MarkaJiap Lucky Colors
      style.textContent = `
        .hljs { color: #1A1625 !important; background: transparent !important; }
        .hljs-keyword, .hljs-type, .hljs-built_in, .hljs-doctag, .hljs-variable.language_ { color: #7B2D8E !important; font-weight: 600; }
        .hljs-title, .hljs-title.class_, .hljs-title.function_ { color: #5A1F6B !important; font-weight: 600; }
        .hljs-string, .hljs-meta .hljs-string, .hljs-regexp { color: #986801 !important; }
        .hljs-number, .hljs-literal { color: #005CC5 !important; }
        .hljs-attr { color: #D63384 !important; }
        .hljs-attribute, .hljs-variable { color: #0D7377 !important; }
        .hljs-symbol, .hljs-bullet { color: #D4A634 !important; }
        .hljs-comment, .hljs-code, .hljs-formula { color: #6A737D !important; font-style: italic; }
        .hljs-name, .hljs-selector-tag, .hljs-quote { color: #22863A !important; }
        .hljs-operator, .hljs-meta { color: #D73A49 !important; }
        .hljs-property { color: #005CC5 !important; }
        .hljs-params { color: #E36209 !important; }
        .hljs-punctuation { color: #6B6B7B !important; }
        .hljs-section { color: #7B2D8E !important; font-weight: 700; }
        .hljs-subst { color: #1A1625 !important; }
        .hljs-addition { color: #22863A !important; background-color: #F0FFF4 !important; }
        .hljs-deletion { color: #B31D28 !important; background-color: #FFEEF0 !important; }
      `;
    }
    
    document.head.appendChild(style);
  }

  // Load content from storage
  async loadContent() {
    return new Promise((resolve) => {
      chrome.storage.local.get(['viewerData'], async (result) => {
        const data = result.viewerData;
        
        if (!data) {
          this.showError('No content to display. Please use the popup to load markdown.');
          resolve();
          return;
        }

        try {
          if (data.type === 'url') {
            // Fetch from URL
            const response = await fetch(data.url);
            if (!response.ok) throw new Error(`Failed to fetch: ${response.status}`);
            this.markdown = await response.text();
            this.updateFileInfo(this.getFilenameFromUrl(data.url));
          } else {
            // Direct text content
            this.markdown = data.content || '';
            this.updateFileInfo(data.filename || 'Untitled.md');
          }

          // Ensure markdown is a string
          if (typeof this.markdown !== 'string') {
            this.markdown = String(this.markdown || '');
          }

          // Render the content
          if (this.markdown.trim()) {
            this.render();
          } else {
            this.showError('Empty content');
          }
          
          // Clear storage
          chrome.storage.local.remove(['viewerData']);
          
        } catch (error) {
          this.showError(error.message);
        }
        
        resolve();
      });
    });
  }

  // Configure marked.js
  configureMarked() {
    // Get marked instance (handle different module formats)
    const markedLib = window.marked || marked;
    
    // Custom renderer
    const renderer = new markedLib.Renderer();
    
    // Add IDs to headings for TOC
    renderer.heading = ({ text, depth }) => {
      const id = this.slugify(text);
      return `<h${depth} id="${id}">${text}</h${depth}>`;
    };

    // Task list checkboxes
    renderer.listitem = ({ text, task, checked }) => {
      if (task) {
        return `<li class="task-list-item">
          <input type="checkbox" ${checked ? 'checked' : ''} disabled>
          ${text}
        </li>`;
      }
      return `<li>${text}</li>`;
    };

    // Code blocks with language label
    renderer.code = ({ text, lang }) => {
      const code = text || '';
      const language = lang || '';
      const validLang = language && hljs.getLanguage(language) ? language : 'plaintext';
      const highlighted = hljs.highlight(code, { language: validLang }).value;
      const langLabel = language || 'code';
      
      return `<pre data-lang="${langLabel}"><code class="hljs language-${validLang}">${highlighted}</code><button class="code-copy-btn" onclick="copyCode(this)">Copy</button></pre>`;
    };

    // Images with loading lazy
    renderer.image = ({ href, title, text }) => {
      const titleAttr = title ? ` title="${title}"` : '';
      return `<img src="${href || ''}" alt="${text || ''}"${titleAttr} loading="lazy">`;
    };

    // Links open in new tab for external
    renderer.link = ({ href, title, text }) => {
      const url = href || '';
      const titleAttr = title ? ` title="${title}"` : '';
      const external = url.startsWith('http') ? ' target="_blank" rel="noopener noreferrer"' : '';
      return `<a href="${url}"${titleAttr}${external}>${text || ''}</a>`;
    };

    markedLib.setOptions({
      renderer,
      gfm: true,
      breaks: true
    });
    
    // Store reference
    this.marked = markedLib;
  }

  // Render markdown
  render() {
    this.configureMarked();
    
    const contentEl = document.getElementById('markdownContent');
    
    // Check if markdown exists
    if (!this.markdown) {
      this.showError('No content to display');
      return;
    }
    
    // Process special blocks before marked
    let processed = this.markdown;
    
    // Extract mermaid blocks
    const mermaidBlocks = [];
    processed = processed.replace(/```mermaid\n([\s\S]*?)```/g, (match, code) => {
      const id = `mermaid-${mermaidBlocks.length}`;
      mermaidBlocks.push({ id, code: code.trim() });
      return `<div class="mermaid" id="${id}"></div>`;
    });

    // Render markdown
    let html = this.marked.parse(processed);
    
    // Sanitize HTML
    html = DOMPurify.sanitize(html, {
      ADD_TAGS: ['iframe'],
      ADD_ATTR: ['target', 'rel', 'loading']
    });
    
    contentEl.innerHTML = html;

    // Process mermaid diagrams
    if (this.options.enableMermaid && mermaidBlocks.length > 0) {
      this.renderMermaid(mermaidBlocks);
    }

    // Process math equations
    if (this.options.enableMath) {
      this.renderMath();
    }

    // Generate TOC
    if (this.options.showToc) {
      this.generateToc();
    }

    // Setup scroll tracking
    this.setupScrollTracking();
    
    // Update word count
    this.updateStats();
  }

  // Render mermaid diagrams
  async renderMermaid(blocks) {
    mermaid.initialize({
      startOnLoad: false,
      theme: this.options.theme === 'dark' ? 'dark' : 'default',
      securityLevel: 'loose',
      fontFamily: 'IBM Plex Sans'
    });

    for (const block of blocks) {
      try {
        const el = document.getElementById(block.id);
        if (el) {
          const { svg } = await mermaid.render(`${block.id}-svg`, block.code);
          el.innerHTML = svg;
        }
      } catch (error) {
        console.error('Mermaid render error:', error);
        const el = document.getElementById(block.id);
        if (el) {
          el.innerHTML = `<pre class="mermaid-error">Diagram error: ${error.message}</pre>`;
        }
      }
    }
  }

  // Render math equations
  renderMath() {
    renderMathInElement(document.getElementById('markdownContent'), {
      delimiters: [
        { left: '$$', right: '$$', display: true },
        { left: '$', right: '$', display: false },
        { left: '\\[', right: '\\]', display: true },
        { left: '\\(', right: '\\)', display: false }
      ],
      throwOnError: false
    });
  }

  // Generate Table of Contents
  generateToc() {
    const contentEl = document.getElementById('markdownContent');
    const tocNav = document.getElementById('tocNav');
    const headings = contentEl.querySelectorAll('h1, h2, h3, h4, h5, h6');
    
    tocNav.innerHTML = '';
    
    headings.forEach((heading) => {
      const level = parseInt(heading.tagName[1]);
      const text = heading.textContent;
      const id = heading.id;
      
      const li = document.createElement('li');
      li.className = 'toc-item';
      
      const a = document.createElement('a');
      a.className = `toc-link toc-h${level}`;
      a.href = `#${id}`;
      a.textContent = text;
      a.addEventListener('click', (e) => {
        e.preventDefault();
        heading.scrollIntoView({ behavior: 'smooth' });
      });
      
      li.appendChild(a);
      tocNav.appendChild(li);
    });
  }

  // Setup UI interactions
  setupUI() {
    // Theme toggle
    document.getElementById('themeToggle').addEventListener('click', () => {
      const current = document.documentElement.getAttribute('data-theme');
      const newTheme = current === 'dark' ? 'light' : 'dark';
      document.documentElement.setAttribute('data-theme', newTheme);
      chrome.storage.sync.set({ theme: newTheme });
      this.options.theme = newTheme;
      this.updateHighlightTheme();
      this.injectHighlightStyles();
      
      // Re-render mermaid with new theme
      if (this.options.enableMermaid) {
        this.render();
      }
    });

    // Sidebar toggle
    document.getElementById('toggleSidebar').addEventListener('click', () => {
      const sidebar = document.getElementById('sidebar');
      sidebar.classList.toggle('collapsed');
    });

    // Search
    this.setupSearch();

    // Export dropdown
    this.setupExportDropdown();

    // Print
    document.getElementById('printBtn').addEventListener('click', () => {
      window.print();
    });

    // Settings
    document.getElementById('settingsBtn').addEventListener('click', () => {
      chrome.runtime.openOptionsPage();
    });

    // Scroll to top
    this.setupScrollToTop();
  }

  // Setup search functionality
  setupSearch() {
    const searchBtn = document.getElementById('searchBtn');
    const searchOverlay = document.getElementById('searchOverlay');
    const searchInput = document.getElementById('searchInput');
    const searchResults = document.getElementById('searchResults');

    searchBtn.addEventListener('click', () => {
      searchOverlay.classList.add('active');
      searchInput.focus();
    });

    searchOverlay.addEventListener('click', (e) => {
      if (e.target === searchOverlay) {
        searchOverlay.classList.remove('active');
      }
    });

    searchInput.addEventListener('input', () => {
      this.performSearch(searchInput.value);
    });

    searchInput.addEventListener('keydown', (e) => {
      if (e.key === 'Escape') {
        searchOverlay.classList.remove('active');
      } else if (e.key === 'Enter') {
        this.jumpToSearchResult(this.currentSearchIndex);
      } else if (e.key === 'ArrowDown') {
        e.preventDefault();
        this.navigateSearchResults(1);
      } else if (e.key === 'ArrowUp') {
        e.preventDefault();
        this.navigateSearchResults(-1);
      }
    });
  }

  performSearch(query) {
    const searchResults = document.getElementById('searchResults');
    
    if (!query.trim()) {
      searchResults.innerHTML = '<div class="search-empty">Type to search...</div>';
      return;
    }

    const content = document.getElementById('markdownContent');
    const text = content.textContent;
    const regex = new RegExp(query, 'gi');
    const matches = [];
    let match;

    while ((match = regex.exec(text)) !== null) {
      const start = Math.max(0, match.index - 40);
      const end = Math.min(text.length, match.index + query.length + 40);
      let preview = text.slice(start, end);
      
      if (start > 0) preview = '...' + preview;
      if (end < text.length) preview = preview + '...';
      
      preview = preview.replace(regex, '<mark>$&</mark>');
      
      matches.push({
        index: match.index,
        preview
      });
    }

    this.searchResults = matches;
    this.currentSearchIndex = 0;

    if (matches.length === 0) {
      searchResults.innerHTML = '<div class="search-empty">No results found</div>';
      return;
    }

    searchResults.innerHTML = matches.map((m, i) => `
      <div class="search-result-item ${i === 0 ? 'selected' : ''}" data-index="${i}">
        <div class="search-result-preview">${m.preview}</div>
      </div>
    `).join('');

    searchResults.querySelectorAll('.search-result-item').forEach(item => {
      item.addEventListener('click', () => {
        this.jumpToSearchResult(parseInt(item.dataset.index));
      });
    });
  }

  navigateSearchResults(direction) {
    const items = document.querySelectorAll('.search-result-item');
    if (items.length === 0) return;

    items[this.currentSearchIndex]?.classList.remove('selected');
    this.currentSearchIndex = (this.currentSearchIndex + direction + items.length) % items.length;
    items[this.currentSearchIndex]?.classList.add('selected');
    items[this.currentSearchIndex]?.scrollIntoView({ block: 'nearest' });
  }

  jumpToSearchResult(index) {
    const searchOverlay = document.getElementById('searchOverlay');
    searchOverlay.classList.remove('active');
    
    // Find and scroll to the text
    // This is a simplified version - could be enhanced with actual highlighting
    const content = document.getElementById('markdownContent');
    const query = document.getElementById('searchInput').value;
    
    if (window.find) {
      window.find(query);
    }
  }

  // Setup export dropdown
  setupExportDropdown() {
    const dropdown = document.getElementById('exportDropdown');
    const btn = document.getElementById('exportBtn');
    const items = dropdown.querySelectorAll('.dropdown-item');

    btn.addEventListener('click', () => {
      dropdown.classList.toggle('open');
    });

    document.addEventListener('click', (e) => {
      if (!dropdown.contains(e.target)) {
        dropdown.classList.remove('open');
      }
    });

    items.forEach(item => {
      item.addEventListener('click', () => {
        const type = item.dataset.export;
        this.handleExport(type);
        dropdown.classList.remove('open');
      });
    });
  }

  handleExport(type) {
    const content = document.getElementById('markdownContent').innerHTML;
    const filename = document.getElementById('fileName').textContent.replace('.md', '');

    switch (type) {
      case 'html':
        this.exportHtml(filename);
        break;
      case 'pdf':
        window.print();
        break;
      case 'copy':
        navigator.clipboard.writeText(content);
        this.showToast('HTML copied to clipboard');
        break;
      case 'copyMd':
        navigator.clipboard.writeText(this.markdown);
        this.showToast('Markdown copied to clipboard');
        break;
    }
  }

  exportHtml(filename) {
    const content = document.getElementById('markdownContent').innerHTML;
    const html = `<!DOCTYPE html>
<html>
<head>
  <meta charset="UTF-8">
  <title>${filename}</title>
  <style>
    body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif; max-width: 800px; margin: 0 auto; padding: 40px; }
    pre { background: #f6f8fa; padding: 16px; border-radius: 8px; overflow-x: auto; }
    code { background: #f6f8fa; padding: 2px 6px; border-radius: 4px; }
    blockquote { border-left: 4px solid #E60012; padding-left: 16px; margin-left: 0; color: #666; }
    table { border-collapse: collapse; width: 100%; }
    th, td { border: 1px solid #ddd; padding: 8px; }
    th { background: #f6f8fa; }
  </style>
</head>
<body>
${content}
</body>
</html>`;

    const blob = new Blob([html], { type: 'text/html' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `${filename}.html`;
    a.click();
    URL.revokeObjectURL(url);
  }

  // Setup scroll to top button
  setupScrollToTop() {
    const btn = document.getElementById('scrollTopBtn');
    
    window.addEventListener('scroll', () => {
      if (window.scrollY > 300) {
        btn.classList.add('visible');
      } else {
        btn.classList.remove('visible');
      }
    });

    btn.addEventListener('click', () => {
      window.scrollTo({ top: 0, behavior: 'smooth' });
    });
  }

  // Setup scroll tracking for TOC
  setupScrollTracking() {
    const headings = document.querySelectorAll('#markdownContent h1, #markdownContent h2, #markdownContent h3, #markdownContent h4, #markdownContent h5, #markdownContent h6');
    const tocLinks = document.querySelectorAll('.toc-link');
    const progressBar = document.getElementById('readingProgress');

    const observer = new IntersectionObserver((entries) => {
      entries.forEach(entry => {
        if (entry.isIntersecting) {
          tocLinks.forEach(link => link.classList.remove('active'));
          const activeLink = document.querySelector(`.toc-link[href="#${entry.target.id}"]`);
          if (activeLink) activeLink.classList.add('active');
        }
      });
    }, { rootMargin: '-20% 0px -80% 0px' });

    headings.forEach(heading => observer.observe(heading));

    // Reading progress
    window.addEventListener('scroll', () => {
      const scrollTop = window.scrollY;
      const docHeight = document.documentElement.scrollHeight - window.innerHeight;
      const progress = (scrollTop / docHeight) * 100;
      progressBar.style.width = `${progress}%`;
    });
  }

  // Keyboard shortcuts
  setupKeyboardShortcuts() {
    document.addEventListener('keydown', (e) => {
      // Ctrl/Cmd + F - Search
      if ((e.ctrlKey || e.metaKey) && e.key === 'f') {
        e.preventDefault();
        document.getElementById('searchBtn').click();
      }
      
      // Ctrl/Cmd + P - Print
      if ((e.ctrlKey || e.metaKey) && e.key === 'p') {
        e.preventDefault();
        window.print();
      }
      
      // Ctrl/Cmd + B - Toggle sidebar
      if ((e.ctrlKey || e.metaKey) && e.key === 'b') {
        e.preventDefault();
        document.getElementById('toggleSidebar').click();
      }

      // Escape - Close overlays
      if (e.key === 'Escape') {
        document.getElementById('searchOverlay').classList.remove('active');
      }
    });
  }

  // Utility methods
  slugify(text) {
    return text
      .toLowerCase()
      .replace(/[^\w\s-]/g, '')
      .replace(/\s+/g, '-')
      .replace(/-+/g, '-')
      .trim();
  }

  getFilenameFromUrl(url) {
    const parts = url.split('/');
    return parts[parts.length - 1] || 'document.md';
  }

  updateFileInfo(filename) {
    document.getElementById('fileName').textContent = filename;
    document.title = `${filename} - JVC MarkView`;
  }

  updateStats() {
    const content = document.getElementById('markdownContent').textContent;
    const words = content.trim().split(/\s+/).filter(w => w.length > 0).length;
    const chars = content.length;
    document.getElementById('fileMeta').textContent = `${words.toLocaleString()} words • ${chars.toLocaleString()} characters`;
  }

  updateHighlightTheme() {
    const isDark = this.options.theme === 'dark';
    document.getElementById('hljs-light').disabled = isDark;
    document.getElementById('hljs-dark').disabled = !isDark;
  }

  showError(message) {
    const content = document.getElementById('markdownContent');
    content.innerHTML = `
      <div style="text-align: center; padding: 60px; color: var(--text-secondary);">
        <svg width="64" height="64" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="margin-bottom: 16px; opacity: 0.5;">
          <circle cx="12" cy="12" r="10"/>
          <line x1="12" y1="8" x2="12" y2="12"/>
          <line x1="12" y1="16" x2="12.01" y2="16"/>
        </svg>
        <h2 style="margin-bottom: 8px; color: var(--text-primary);">Unable to Load Content</h2>
        <p>${message}</p>
      </div>
    `;
  }

  showToast(message) {
    const toast = document.createElement('div');
    toast.style.cssText = `
      position: fixed;
      bottom: 24px;
      left: 50%;
      transform: translateX(-50%);
      background: var(--accent);
      color: white;
      padding: 12px 24px;
      border-radius: 8px;
      font-size: 14px;
      font-weight: 500;
      z-index: 9999;
      animation: fadeIn 0.2s ease;
    `;
    toast.textContent = message;
    document.body.appendChild(toast);
    
    setTimeout(() => {
      toast.style.animation = 'fadeOut 0.2s ease';
      setTimeout(() => toast.remove(), 200);
    }, 2000);
  }
}

// Copy code function (global for onclick)
function copyCode(btn) {
  const pre = btn.closest('pre');
  const code = pre.querySelector('code').textContent;
  navigator.clipboard.writeText(code);
  
  const originalText = btn.textContent;
  btn.textContent = 'Copied!';
  btn.style.background = 'var(--accent)';
  btn.style.color = 'white';
  
  setTimeout(() => {
    btn.textContent = originalText;
    btn.style.background = '';
    btn.style.color = '';
  }, 2000);
}

// Initialize viewer
document.addEventListener('DOMContentLoaded', () => {
  new MarkViewViewer();
});
