// JVC MarkView - Popup Script

class MarkViewPopup {
  constructor() {
    this.selectedFile = null;
    this.init();
  }

  init() {
    this.setupTheme();
    this.setupTabs();
    this.setupFileHandling();
    this.setupUrlInput();
    this.setupButtons();
    this.setupFooter();
  }

  // Theme handling
  setupTheme() {
    const themeToggle = document.getElementById('themeToggle');
    
    // Load saved theme
    chrome.storage.sync.get(['theme'], (result) => {
      const theme = result.theme || 'light';
      document.documentElement.setAttribute('data-theme', theme);
    });

    themeToggle.addEventListener('click', () => {
      const current = document.documentElement.getAttribute('data-theme');
      const newTheme = current === 'dark' ? 'light' : 'dark';
      document.documentElement.setAttribute('data-theme', newTheme);
      chrome.storage.sync.set({ theme: newTheme });
    });
  }

  // Tab switching
  setupTabs() {
    const tabs = document.querySelectorAll('.tab');
    const contents = document.querySelectorAll('.tab-content');

    tabs.forEach(tab => {
      tab.addEventListener('click', () => {
        const tabId = tab.dataset.tab;
        
        tabs.forEach(t => t.classList.remove('active'));
        contents.forEach(c => c.classList.remove('active'));
        
        tab.classList.add('active');
        document.getElementById(`${tabId}Tab`).classList.add('active');
      });
    });
  }

  // File handling (drag & drop, browse)
  setupFileHandling() {
    const dropZone = document.getElementById('fileDropZone');
    const fileInput = document.getElementById('fileInput');
    const browseBtn = document.getElementById('browseBtn');
    const selectedFileEl = document.getElementById('selectedFile');
    const fileNameEl = document.getElementById('fileName');
    const fileSizeEl = document.getElementById('fileSize');
    const removeBtn = document.getElementById('removeFile');
    const renderBtn = document.getElementById('renderFile');

    // Click to browse
    browseBtn.addEventListener('click', (e) => {
      e.stopPropagation();
      fileInput.click();
    });

    dropZone.addEventListener('click', () => {
      fileInput.click();
    });

    // File input change
    fileInput.addEventListener('change', (e) => {
      if (e.target.files.length > 0) {
        this.handleFile(e.target.files[0]);
      }
    });

    // Drag and drop
    dropZone.addEventListener('dragover', (e) => {
      e.preventDefault();
      dropZone.classList.add('dragover');
    });

    dropZone.addEventListener('dragleave', () => {
      dropZone.classList.remove('dragover');
    });

    dropZone.addEventListener('drop', (e) => {
      e.preventDefault();
      dropZone.classList.remove('dragover');
      
      if (e.dataTransfer.files.length > 0) {
        this.handleFile(e.dataTransfer.files[0]);
      }
    });

    // Remove file
    removeBtn.addEventListener('click', () => {
      this.selectedFile = null;
      dropZone.hidden = false;
      selectedFileEl.hidden = true;
      renderBtn.disabled = true;
      fileInput.value = '';
    });
  }

  handleFile(file) {
    // Check file type
    const validTypes = ['.md', '.markdown', '.txt'];
    const ext = '.' + file.name.split('.').pop().toLowerCase();
    
    if (!validTypes.includes(ext)) {
      alert('Please select a Markdown file (.md, .markdown, or .txt)');
      return;
    }

    this.selectedFile = file;
    
    // Update UI
    const dropZone = document.getElementById('fileDropZone');
    const selectedFileEl = document.getElementById('selectedFile');
    const fileNameEl = document.getElementById('fileName');
    const fileSizeEl = document.getElementById('fileSize');
    const renderBtn = document.getElementById('renderFile');

    dropZone.hidden = true;
    selectedFileEl.hidden = false;
    fileNameEl.textContent = file.name;
    fileSizeEl.textContent = this.formatFileSize(file.size);
    renderBtn.disabled = false;
  }

  formatFileSize(bytes) {
    if (bytes < 1024) return bytes + ' B';
    if (bytes < 1024 * 1024) return (bytes / 1024).toFixed(1) + ' KB';
    return (bytes / (1024 * 1024)).toFixed(1) + ' MB';
  }

  // URL input handling
  setupUrlInput() {
    const urlInput = document.getElementById('urlInput');
    const renderBtn = document.getElementById('renderUrl');
    const chips = document.querySelectorAll('.chip');

    urlInput.addEventListener('input', () => {
      renderBtn.disabled = !this.isValidUrl(urlInput.value);
    });

    urlInput.addEventListener('keypress', (e) => {
      if (e.key === 'Enter' && !renderBtn.disabled) {
        this.openViewer({ type: 'url', url: urlInput.value });
      }
    });

    // Quick link chips
    chips.forEach(chip => {
      chip.addEventListener('click', () => {
        const urlType = chip.dataset.url;
        if (urlType === 'github') {
          urlInput.value = 'https://raw.githubusercontent.com/';
        } else if (urlType === 'gitlab') {
          urlInput.value = 'https://gitlab.com/';
        }
        urlInput.focus();
        renderBtn.disabled = !this.isValidUrl(urlInput.value);
      });
    });
  }

  isValidUrl(string) {
    try {
      new URL(string);
      return true;
    } catch {
      return false;
    }
  }

  // Button handlers
  setupButtons() {
    // Render paste
    document.getElementById('renderPaste').addEventListener('click', () => {
      const content = document.getElementById('markdownInput').value;
      if (content.trim()) {
        this.openViewer({ type: 'text', content });
      }
    });

    // Render file
    document.getElementById('renderFile').addEventListener('click', () => {
      if (this.selectedFile) {
        const reader = new FileReader();
        reader.onload = (e) => {
          this.openViewer({ 
            type: 'text', 
            content: e.target.result,
            filename: this.selectedFile.name
          });
        };
        reader.readAsText(this.selectedFile);
      }
    });

    // Render URL
    document.getElementById('renderUrl').addEventListener('click', () => {
      const url = document.getElementById('urlInput').value;
      if (this.isValidUrl(url)) {
        this.openViewer({ type: 'url', url });
      }
    });
  }

  // Footer buttons
  setupFooter() {
    document.getElementById('openOptions').addEventListener('click', () => {
      chrome.runtime.openOptionsPage();
    });

    document.getElementById('openHelp').addEventListener('click', () => {
      chrome.tabs.create({ url: 'options.html#help' });
    });
  }

  // Open viewer
  openViewer(data) {
    chrome.runtime.sendMessage({ 
      action: 'openViewer', 
      data 
    }, () => {
      window.close();
    });
  }
}

// Initialize popup
document.addEventListener('DOMContentLoaded', () => {
  new MarkViewPopup();
});
