// JVC MarkView - Options Script

class MarkViewOptions {
  constructor() {
    this.options = {};
    this.init();
  }

  async init() {
    await this.loadOptions();
    this.setupNavigation();
    this.setupTheme();
    this.setupToggles();
    this.setupRangeInputs();
    this.setupCustomCss();
    this.handleHashChange();
  }

  // Load saved options
  async loadOptions() {
    return new Promise((resolve) => {
      chrome.storage.sync.get(null, (result) => {
        this.options = {
          theme: result.theme || 'auto',
          autoRender: result.autoRender !== false,
          showToc: result.showToc !== false,
          codeLineNumbers: result.codeLineNumbers !== false,
          enableMermaid: result.enableMermaid !== false,
          enableMath: result.enableMath !== false,
          enableHighlight: result.enableHighlight !== false,
          codeTheme: result.codeTheme || 'markajiap',
          enableTaskLists: result.enableTaskLists !== false,
          autoRefresh: result.autoRefresh !== false,
          autoRefreshInterval: result.autoRefreshInterval || 2000,
          fontSize: result.fontSize || 16,
          fontFamily: result.fontFamily || 'ibm-plex',
          lineHeight: result.lineHeight || 1.6,
          contentWidth: result.contentWidth || 900,
          customCss: result.customCss || ''
        };

        // Apply to UI
        this.applyOptionsToUI();
        
        // Apply theme
        document.documentElement.setAttribute('data-theme', this.options.theme);
        
        resolve();
      });
    });
  }

  // Apply options to form elements
  applyOptionsToUI() {
    // Toggles
    document.getElementById('autoRender').checked = this.options.autoRender;
    document.getElementById('showToc').checked = this.options.showToc;
    document.getElementById('codeLineNumbers').checked = this.options.codeLineNumbers;
    document.getElementById('enableMermaid').checked = this.options.enableMermaid;
    document.getElementById('enableMath').checked = this.options.enableMath;
    document.getElementById('enableHighlight').checked = this.options.enableHighlight;
    document.getElementById('codeTheme').value = this.options.codeTheme;
    document.getElementById('enableTaskLists').checked = this.options.enableTaskLists;
    document.getElementById('autoRefresh').checked = this.options.autoRefresh;
    document.getElementById('autoRefreshInterval').value = this.options.autoRefreshInterval;

    // Range inputs
    document.getElementById('fontSize').value = this.options.fontSize;
    document.getElementById('fontSizeValue').textContent = this.options.fontSize + 'px';
    
    document.getElementById('fontFamily').value = this.options.fontFamily;
    
    document.getElementById('lineHeight').value = this.options.lineHeight;
    document.getElementById('lineHeightValue').textContent = this.options.lineHeight;
    
    document.getElementById('contentWidth').value = this.options.contentWidth;
    document.getElementById('contentWidthValue').textContent = this.options.contentWidth + 'px';

    // Theme buttons
    document.querySelectorAll('.theme-btn').forEach(btn => {
      btn.classList.toggle('active', btn.dataset.theme === this.options.theme);
    });

    // Custom CSS
    document.getElementById('customCss').value = this.options.customCss;
  }

  // Save options
  saveOptions() {
    chrome.storage.sync.set(this.options, () => {
      this.showToast('Settings saved');
    });
  }

  // Setup sidebar navigation
  setupNavigation() {
    const navItems = document.querySelectorAll('.nav-item');
    const sections = document.querySelectorAll('.options-section');

    navItems.forEach(item => {
      item.addEventListener('click', (e) => {
        e.preventDefault();
        const targetId = item.getAttribute('href').slice(1);
        
        // Update nav
        navItems.forEach(n => n.classList.remove('active'));
        item.classList.add('active');
        
        // Update sections
        sections.forEach(s => s.classList.remove('active'));
        document.getElementById(targetId).classList.add('active');
        
        // Update hash
        history.pushState(null, '', `#${targetId}`);
      });
    });
  }

  // Handle initial hash
  handleHashChange() {
    const hash = window.location.hash.slice(1);
    if (hash) {
      const navItem = document.querySelector(`.nav-item[href="#${hash}"]`);
      if (navItem) {
        navItem.click();
      }
    }
  }

  // Setup theme selection
  setupTheme() {
    const themeBtns = document.querySelectorAll('.theme-btn');
    
    themeBtns.forEach(btn => {
      btn.addEventListener('click', () => {
        const theme = btn.dataset.theme;
        
        // Update UI
        themeBtns.forEach(b => b.classList.remove('active'));
        btn.classList.add('active');
        
        // Apply theme
        document.documentElement.setAttribute('data-theme', theme);
        
        // Save
        this.options.theme = theme;
        this.saveOptions();
      });
    });
  }

  // Setup toggle switches
  setupToggles() {
    const toggles = [
      'autoRender',
      'showToc', 
      'codeLineNumbers',
      'enableMermaid',
      'enableMath',
      'enableHighlight',
      'enableTaskLists',
      'autoRefresh'
    ];

    toggles.forEach(id => {
      const toggle = document.getElementById(id);
      if (toggle) {
        toggle.addEventListener('change', () => {
          this.options[id] = toggle.checked;
          this.saveOptions();
        });
      }
    });

    // Auto-refresh interval select
    const intervalSelect = document.getElementById('autoRefreshInterval');
    if (intervalSelect) {
      intervalSelect.addEventListener('change', () => {
        this.options.autoRefreshInterval = parseInt(intervalSelect.value);
        this.saveOptions();
      });
    }

    // Code theme select
    const codeThemeSelect = document.getElementById('codeTheme');
    if (codeThemeSelect) {
      codeThemeSelect.addEventListener('change', () => {
        this.options.codeTheme = codeThemeSelect.value;
        this.saveOptions();
      });
    }
  }

  // Setup range inputs
  setupRangeInputs() {
    // Font size
    const fontSize = document.getElementById('fontSize');
    const fontSizeValue = document.getElementById('fontSizeValue');
    
    fontSize.addEventListener('input', () => {
      fontSizeValue.textContent = fontSize.value + 'px';
    });
    
    fontSize.addEventListener('change', () => {
      this.options.fontSize = parseInt(fontSize.value);
      this.saveOptions();
    });

    // Content width
    const contentWidth = document.getElementById('contentWidth');
    const contentWidthValue = document.getElementById('contentWidthValue');
    
    contentWidth.addEventListener('input', () => {
      contentWidthValue.textContent = contentWidth.value + 'px';
    });
    
    contentWidth.addEventListener('change', () => {
      this.options.contentWidth = parseInt(contentWidth.value);
      this.saveOptions();
    });

    // Font family select
    const fontFamily = document.getElementById('fontFamily');
    if (fontFamily) {
      fontFamily.addEventListener('change', () => {
        this.options.fontFamily = fontFamily.value;
        this.saveOptions();
      });
    }

    // Line height
    const lineHeight = document.getElementById('lineHeight');
    const lineHeightValue = document.getElementById('lineHeightValue');
    
    if (lineHeight) {
      lineHeight.addEventListener('input', () => {
        lineHeightValue.textContent = lineHeight.value;
      });
      
      lineHeight.addEventListener('change', () => {
        this.options.lineHeight = parseFloat(lineHeight.value);
        this.saveOptions();
      });
    }
  }

  // Setup custom CSS editor
  setupCustomCss() {
    const textarea = document.getElementById('customCss');
    const saveBtn = document.getElementById('saveCss');
    const resetBtn = document.getElementById('resetCss');

    saveBtn.addEventListener('click', () => {
      this.options.customCss = textarea.value;
      this.saveOptions();
    });

    resetBtn.addEventListener('click', () => {
      textarea.value = '';
      this.options.customCss = '';
      this.saveOptions();
    });

    // Auto-save on blur
    textarea.addEventListener('blur', () => {
      if (textarea.value !== this.options.customCss) {
        this.options.customCss = textarea.value;
        this.saveOptions();
      }
    });
  }

  // Show toast notification
  showToast(message) {
    // Remove existing toast
    const existing = document.querySelector('.toast');
    if (existing) existing.remove();

    const toast = document.createElement('div');
    toast.className = 'toast';
    toast.textContent = message;
    document.body.appendChild(toast);

    setTimeout(() => {
      toast.style.animation = 'slideOut 0.3s ease forwards';
      setTimeout(() => toast.remove(), 300);
    }, 2000);
  }
}

// Initialize
document.addEventListener('DOMContentLoaded', () => {
  new MarkViewOptions();
});
