// MarkaJiap Mermaid Renderer
// This file runs in page context to render mermaid diagrams

(function() {
  console.log('🎨 MarkaJiap [mermaid-render.js]: Starting...');
  
  // Get config from DOM
  const configEl = document.getElementById('markajiap-mermaid-config');
  let isDark = false;
  
  if (configEl) {
    isDark = configEl.dataset.dark === 'true';
    console.log('🎨 MarkaJiap: Config found, dark mode:', isDark);
  }
  
  // Clean mermaid syntax - remove unsupported features and convert to simpler diagrams
  function cleanMermaidSyntax(code) {
    let cleaned = code;
    
    // Check if it's an architecture diagram (not supported in older mermaid)
    if (/^architecture/im.test(cleaned)) {
      console.log('🔧 MarkaJiap: Detected architecture diagram, converting to flowchart...');
      cleaned = convertArchitectureToFlowchart(cleaned);
      return cleaned;
    }
    
    // Remove @{ icon: "...", ... } syntax and convert to simple node
    // Pattern: NodeName@{ icon: "...", pos: "..." }  → NodeName
    cleaned = cleaned.replace(/(\w+)@\{[^}]*\}/g, '$1');
    
    // Remove @{ shape: "...", ... } syntax
    cleaned = cleaned.replace(/@\{[^}]*\}/g, '');
    
    // Remove icon definitions: %%{ icon: ... }%%
    cleaned = cleaned.replace(/%%\{[^}]*\}%%/g, '');
    
    // Clean up any double spaces
    cleaned = cleaned.replace(/  +/g, ' ');
    
    // Clean up consecutive newlines
    cleaned = cleaned.replace(/\n\s*\n\s*\n/g, '\n\n');
    
    return cleaned;
  }
  
  // Convert architecture diagram to simple flowchart
  function convertArchitectureToFlowchart(code) {
    let lines = code.split('\n');
    let nodes = [];
    let connections = [];
    
    // Start with flowchart
    let result = ['flowchart LR'];
    
    for (let i = 0; i < lines.length; i++) {
      let line = lines[i].trim();
      
      // Skip empty lines and comments
      if (!line || line.startsWith('%') || line.startsWith('architecture')) {
        continue;
      }
      
      // Skip group/service/junction definitions (keep node names only)
      // Pattern: group name[Label]
      let groupMatch = line.match(/group\s+(\w+)\[([^\]]*)\]/);
      if (groupMatch) {
        nodes.push({ id: groupMatch[1], label: groupMatch[2] || groupMatch[1] });
        continue;
      }
      
      // Pattern: service name[Label]
      let serviceMatch = line.match(/service\s+(\w+)\[([^\]]*)\]/);
      if (serviceMatch) {
        let id = serviceMatch[1];
        let label = serviceMatch[2] || serviceMatch[1];
        // Remove @{ ... } from label
        label = label.replace(/@\{[^}]*\}/g, '').trim();
        if (label) {
          nodes.push({ id: id, label: label });
        }
        continue;
      }
      
      // Pattern: nodeA --> nodeB or nodeA:side --> nodeB:side
      let connMatch = line.match(/(\w+)(?::\w+)?\s*(-->|<-->|---)\s*(\w+)(?::\w+)?/);
      if (connMatch) {
        connections.push({ from: connMatch[1], arrow: '-->', to: connMatch[3] });
        continue;
      }
      
      // Simple node definition: nodeName[label] (without service/group prefix)
      let nodeMatch = line.match(/^(\w+)\[([^\]]*)\]/);
      if (nodeMatch) {
        let label = nodeMatch[2].replace(/@\{[^}]*\}/g, '').trim();
        if (label) {
          nodes.push({ id: nodeMatch[1], label: label });
        }
        continue;
      }
    }
    
    // Build flowchart
    // Add unique nodes
    let addedNodes = {};
    for (let node of nodes) {
      if (!addedNodes[node.id]) {
        result.push('    ' + node.id + '[' + node.label + ']');
        addedNodes[node.id] = true;
      }
    }
    
    // Add connections
    for (let conn of connections) {
      result.push('    ' + conn.from + ' ' + conn.arrow + ' ' + conn.to);
    }
    
    console.log('🔧 MarkaJiap: Converted to flowchart with', nodes.length, 'nodes and', connections.length, 'connections');
    return result.join('\n');
  }
  
  // Wait for mermaid to be available
  function waitForMermaid(callback, attempts) {
    attempts = attempts || 0;
    
    if (typeof mermaid !== 'undefined' && mermaid.initialize) {
      console.log('✅ MarkaJiap: mermaid is available');
      callback();
    } else if (attempts < 50) {
      setTimeout(function() { waitForMermaid(callback, attempts + 1); }, 100);
    } else {
      console.error('❌ MarkaJiap: mermaid not available after 5s');
    }
  }
  
  waitForMermaid(function() {
    try {
      // Initialize with MarkaJiap theme (lucky colors)
      mermaid.initialize({
        startOnLoad: false,
        theme: 'base',
        securityLevel: 'loose',
        themeVariables: {
          // Primary - Purple (โชคลาภ)
          primaryColor: '#7B2D8E',
          primaryTextColor: '#FFFFFF',
          primaryBorderColor: '#5E2270',
          
          // Secondary - Gold (บารมี)
          secondaryColor: '#D4A634',
          secondaryTextColor: '#1A1625',
          secondaryBorderColor: '#B8922D',
          
          // Tertiary
          tertiaryColor: '#9B3DAE',
          tertiaryTextColor: '#FFFFFF',
          tertiaryBorderColor: '#7B2D8E',
          
          // Background
          background: isDark ? '#16121F' : '#FAF8FC',
          mainBkg: isDark ? '#1F1A2E' : '#FFFFFF',
          
          // Nodes
          nodeBkg: isDark ? '#2A243A' : '#FAF8FC',
          nodeBorder: '#7B2D8E',
          
          // Text & Lines
          textColor: isDark ? '#F0EDF5' : '#1A1625',
          lineColor: '#7B2D8E',
          
          // Flowchart
          clusterBkg: isDark ? '#1F1A2E' : '#EDE9F0',
          clusterBorder: '#7B2D8E',
          
          // Sequence
          actorBkg: '#7B2D8E',
          actorBorder: '#5E2270',
          actorTextColor: '#FFFFFF',
          actorLineColor: '#7B2D8E',
          signalColor: '#D4A634',
          signalTextColor: isDark ? '#F0EDF5' : '#1A1625',
          
          // Notes
          noteBkgColor: '#D4A634',
          noteTextColor: '#1A1625',
          noteBorderColor: '#B8922D',
          
          // Labels
          labelColor: isDark ? '#F0EDF5' : '#1A1625',
          labelTextColor: isDark ? '#F0EDF5' : '#1A1625',
          edgeLabelBackground: isDark ? '#1F1A2E' : '#FFFFFF',
          
          // Pie
          pie1: '#7B2D8E',
          pie2: '#D4A634',
          pie3: '#9B3DAE',
          pie4: '#F4D675',
          pie5: '#5E2270',
          pie6: '#B8922D',
          
          // Font
          fontFamily: '"IBM Plex Sans", -apple-system, sans-serif',
          fontSize: '14px'
        },
        flowchart: {
          useMaxWidth: true,
          htmlLabels: true,
          curve: 'basis'
        }
      });
      
      console.log('✅ MarkaJiap: mermaid.initialize() completed');
      
      // Render all diagrams
      var diagrams = document.querySelectorAll('.mermaid');
      console.log('🖼️ MarkaJiap: Found', diagrams.length, 'diagrams to render');
      
      diagrams.forEach(function(el, i) {
        var code = el.textContent.trim();
        
        // Skip if empty or already rendered
        if (!code || el.querySelector('svg')) {
          console.log('⏭️ MarkaJiap: Skipping diagram', i);
          return;
        }
        
        var id = 'mermaid-' + i + '-' + Date.now();
        console.log('🖼️ MarkaJiap: Rendering diagram', i, 'with id:', id);
        
        // Try to render, with fallback to cleaned syntax
        function tryRender(codeToRender, isRetry) {
          mermaid.render(id + (isRetry ? '-retry' : ''), codeToRender)
            .then(function(result) {
              el.innerHTML = result.svg;
              if (isRetry) {
                console.log('✅ MarkaJiap: Diagram', i, 'rendered with cleaned syntax');
              } else {
                console.log('✅ MarkaJiap: Diagram', i, 'rendered successfully');
              }
            })
            .catch(function(err) {
              if (!isRetry) {
                // First attempt failed, try with cleaned syntax
                console.log('⚠️ MarkaJiap: Diagram', i, 'failed, trying with cleaned syntax...');
                var cleanedCode = cleanMermaidSyntax(code);
                if (cleanedCode !== code) {
                  console.log('🔧 MarkaJiap: Cleaned code:', cleanedCode.substring(0, 100) + '...');
                  tryRender(cleanedCode, true);
                } else {
                  showError(err);
                }
              } else {
                showError(err);
              }
            });
        }
        
        function showError(err) {
          console.error('❌ MarkaJiap: Diagram', i, 'error:', err);
          el.innerHTML = '<div style="color:#E63946;padding:16px;background:' + (isDark ? '#1F1A2E' : '#FFF5F5') + ';border-radius:8px;border:1px solid #E63946;text-align:left;">' +
            '<strong>❌ Mermaid Error</strong><br><pre style="margin-top:8px;font-size:12px;white-space:pre-wrap;">' + 
            (err.message || String(err)) + '</pre></div>';
        }
        
        tryRender(code, false);
      });
      
    } catch (err) {
      console.error('❌ MarkaJiap: Fatal error:', err);
    }
  });
})();

