// MarkaJiap - Content Script
// In-place Markdown Rendering

console.log('🔮 MarkaJiap: Content script loaded');

class MarkaJiapContent {
  constructor() {
    console.log('🔮 MarkaJiap: Constructor called');
    this.isRendered = false;
    this.originalContent = '';
    this.originalTitle = '';
    this.currentMarkdown = '';
    this.contentHash = '';
    this.autoRefreshInterval = null;
    this.autoRefreshEnabled = false;
    this.init();
  }

  init() {
    console.log('🔮 MarkaJiap: init() called');
    console.log('🔮 MarkaJiap: URL:', window.location.href);
    console.log('🔮 MarkaJiap: contentType:', document.contentType);
    
    const isMarkdown = this.isMarkdownFile();
    console.log('🔮 MarkaJiap: isMarkdownFile():', isMarkdown);
    
    if (isMarkdown) {
      console.log('🔮 MarkaJiap: This is a markdown file, showing button and auto-rendering');
      this.showFloatingButton();
      this.checkAutoRender();
    } else {
      console.log('🔮 MarkaJiap: Not a markdown file, skipping');
    }

    chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
      console.log('🔮 MarkaJiap: Received message:', request);
      if (request.action === 'toggleViewer' || request.action === 'autoRender') {
        if (!this.isRendered) {
          this.renderInPlace();
        } else {
          this.restoreOriginal();
        }
        sendResponse({ success: true });
      }
      return true;
    });
  }

  isMarkdownFile() {
    const url = window.location.href.toLowerCase();
    return (
      url.endsWith('.md') ||
      url.endsWith('.markdown') ||
      url.includes('raw.githubusercontent.com') ||
      url.includes('gist.githubusercontent.com') ||
      document.contentType === 'text/markdown' ||
      document.contentType === 'text/x-markdown' ||
      document.contentType === 'text/plain'
    );
  }

  checkAutoRender() {
    console.log('🔮 MarkaJiap: checkAutoRender() called');
    chrome.storage.sync.get(['autoRender'], (result) => {
      console.log('🔮 MarkaJiap: autoRender setting:', result.autoRender);
      if (result.autoRender !== false) {
        console.log('🔮 MarkaJiap: Auto-rendering in 150ms...');
        setTimeout(() => this.renderInPlace(), 150);
      }
    });
  }

  showFloatingButton() {
    const existing = document.getElementById('markajiap-btn');
    if (existing) existing.remove();

    const button = document.createElement('button');
    button.id = 'markajiap-btn';
    button.innerHTML = `
      <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
        <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
        <path d="M14 2v6h6"/>
        <path d="M16 13H8"/><path d="M16 17H8"/><path d="M10 9H8"/>
      </svg>
      <span>MarkaJiap</span>
    `;

    button.style.cssText = `
      position: fixed;
      bottom: 20px;
      right: 20px;
      display: flex;
      align-items: center;
      gap: 8px;
      padding: 12px 16px;
      background: linear-gradient(135deg, #7B2D8E 0%, #5E2270 100%);
      color: white;
      border: none;
      border-radius: 8px;
      font-family: -apple-system, BlinkMacSystemFont, sans-serif;
      font-size: 14px;
      font-weight: 600;
      cursor: pointer;
      box-shadow: 0 4px 12px rgba(123, 45, 142, 0.4);
      z-index: 999999;
      transition: all 0.2s ease;
    `;

    button.addEventListener('mouseenter', () => {
      button.style.transform = 'translateY(-2px)';
      button.style.boxShadow = '0 6px 16px rgba(123, 45, 142, 0.5)';
    });

    button.addEventListener('mouseleave', () => {
      button.style.transform = '';
      button.style.boxShadow = '0 4px 12px rgba(123, 45, 142, 0.4)';
    });

    button.addEventListener('click', () => {
      if (this.isRendered) {
        this.restoreOriginal();
      } else {
        this.renderInPlace();
      }
    });

    document.body.appendChild(button);
  }

  updateButton(isRendered) {
    const button = document.getElementById('markajiap-btn');
    if (!button) return;

    if (isRendered) {
      button.innerHTML = `
        <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
          <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
          <path d="M14 2v6h6"/>
        </svg>
        <span>View Raw</span>
      `;
      button.style.background = 'linear-gradient(135deg, #D4A634 0%, #B8922D 100%)';
    } else {
      button.innerHTML = `
        <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
          <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
          <path d="M14 2v6h6"/>
          <path d="M16 13H8"/><path d="M16 17H8"/><path d="M10 9H8"/>
        </svg>
        <span>MarkaJiap</span>
      `;
      button.style.background = 'linear-gradient(135deg, #7B2D8E 0%, #5E2270 100%)';
    }
  }

  async renderInPlace() {
    console.log('🚀 MarkaJiap: renderInPlace() called');
    
    if (this.isRendered) {
      console.log('🚀 MarkaJiap: Already rendered, skipping');
      return;
    }

    this.originalContent = document.body.innerHTML;
    this.originalTitle = document.title;

    let markdown = '';
    const pre = document.querySelector('pre');
    console.log('🚀 MarkaJiap: Found <pre> element:', !!pre);
    
    if (pre) {
      markdown = pre.textContent;
    } else {
      markdown = document.body.innerText;
    }

    console.log('🚀 MarkaJiap: Markdown content length:', markdown.length);
    console.log('🚀 MarkaJiap: First 200 chars:', markdown.substring(0, 200));

    if (!markdown.trim()) {
      console.error('❌ MarkaJiap: No content found');
      return;
    }

    const options = await this.getOptions();
    console.log('🚀 MarkaJiap: Options:', options);
    
    // Store markdown for auto-refresh
    this.currentMarkdown = markdown;
    
    this.createRenderedView(markdown, options);
    this.isRendered = true;
    this.updateButton(true);
    
    console.log('✅ MarkaJiap: renderInPlace() completed');
  }

  getOptions() {
    return new Promise((resolve) => {
      chrome.storage.sync.get(null, (result) => {
        resolve({
          theme: result.theme || 'auto',  // Default to auto
          showToc: result.showToc !== false,
          fontSize: result.fontSize || 16,
          fontFamily: result.fontFamily || 'ibm-plex',
          lineHeight: result.lineHeight || 1.6,
          customCss: result.customCss || '',
          autoRefresh: result.autoRefresh !== false,
          autoRefreshInterval: result.autoRefreshInterval || 2000,
          codeTheme: result.codeTheme || 'markajiap'
        });
      });
    });
  }

  // Get effective theme (resolve 'auto' to system preference)
  getEffectiveTheme(theme) {
    if (theme === 'auto') {
      const prefersDark = window.matchMedia('(prefers-color-scheme: dark)').matches;
      console.log('🌓 MarkaJiap: System prefers dark mode:', prefersDark);
      return prefersDark ? 'dark' : 'light';
    }
    return theme;
  }

  // Setup listener for system theme changes
  setupThemeListener() {
    const mediaQuery = window.matchMedia('(prefers-color-scheme: dark)');
    
    mediaQuery.addEventListener('change', async (e) => {
      const options = await this.getOptions();
      
      // Only react if theme is set to 'auto'
      if (options.theme === 'auto') {
        const newTheme = e.matches ? 'dark' : 'light';
        console.log('🌓 MarkaJiap: System theme changed to:', newTheme);
        document.documentElement.setAttribute('data-theme', newTheme);
      }
    });
    
    console.log('🌓 MarkaJiap: Theme listener setup complete');
  }

  // Simple hash function for change detection
  simpleHash(str) {
    let hash = 0;
    for (let i = 0; i < str.length; i++) {
      const char = str.charCodeAt(i);
      hash = ((hash << 5) - hash) + char;
      hash = hash & hash; // Convert to 32bit integer
    }
    return hash.toString();
  }

  // Start auto-refresh file watcher
  async startAutoRefresh() {
    const options = await this.getOptions();
    
    if (!options.autoRefresh) {
      console.log('🔄 MarkaJiap: Auto-refresh disabled');
      return;
    }

    if (this.autoRefreshInterval) {
      clearInterval(this.autoRefreshInterval);
    }

    console.log('🔄 MarkaJiap: Starting auto-refresh, interval:', options.autoRefreshInterval, 'ms');
    this.autoRefreshEnabled = true;
    
    this.autoRefreshInterval = setInterval(() => {
      this.checkForChanges();
    }, options.autoRefreshInterval);
    
    // Show indicator
    this.showAutoRefreshIndicator(true);
  }

  // Stop auto-refresh
  stopAutoRefresh() {
    if (this.autoRefreshInterval) {
      clearInterval(this.autoRefreshInterval);
      this.autoRefreshInterval = null;
    }
    this.autoRefreshEnabled = false;
    this.showAutoRefreshIndicator(false);
    console.log('🔄 MarkaJiap: Auto-refresh stopped');
  }

  // Check for file changes
  async checkForChanges() {
    if (!this.isRendered) return;

    // Skip for file:// URLs - CORS blocks fetch on local files
    if (window.location.protocol === 'file:') {
      // Only log once
      if (!this._localFileWarningShown) {
        console.log('🔄 MarkaJiap: Auto-refresh disabled for local files (CORS limitation)');
        this._localFileWarningShown = true;
        // Stop the interval for local files
        this.stopAutoRefresh();
        // Hide the indicator since it won't work
        this.showAutoRefreshIndicator(false);
      }
      return;
    }

    try {
      // For http/https URLs, re-fetch the content
      const response = await fetch(window.location.href, { cache: 'no-store' });
      const text = await response.text();
      
      // Parse markdown from response (might be wrapped in HTML)
      let markdown = text;
      const parser = new DOMParser();
      const doc = parser.parseFromString(text, 'text/html');
      const pre = doc.querySelector('pre');
      if (pre) {
        markdown = pre.textContent;
      }
      
      const newHash = this.simpleHash(markdown);
      
      if (newHash !== this.contentHash && this.contentHash !== '') {
        console.log('🔄 MarkaJiap: File changed, refreshing...');
        this.refreshContent(markdown);
      }
      
      this.contentHash = newHash;
    } catch (err) {
      console.log('🔄 MarkaJiap: Could not check for changes:', err.message);
    }
  }

  // Refresh content without full re-render
  async refreshContent(markdown) {
    this.currentMarkdown = markdown;
    
    const contentEl = document.getElementById('markajiap-content');
    if (!contentEl) return;

    const options = await this.getOptions();
    
    // Store raw markdown for the renderer
    contentEl.setAttribute('data-raw-markdown', markdown);
    
    // Dispatch event to trigger re-render
    window.dispatchEvent(new CustomEvent('markajiap-refresh-content'));
    
    // Flash the refresh indicator
    this.flashRefreshIndicator();
  }

  // Show/hide auto-refresh indicator in toolbar
  showAutoRefreshIndicator(show) {
    let indicator = document.getElementById('markajiap-refresh-indicator');
    
    if (show && !indicator) {
      indicator = document.createElement('span');
      indicator.id = 'markajiap-refresh-indicator';
      indicator.innerHTML = '🔄';
      indicator.title = 'Auto-refresh enabled';
      indicator.style.cssText = `
        display: inline-flex;
        align-items: center;
        margin-left: 8px;
        font-size: 14px;
        opacity: 0.7;
        animation: spin 2s linear infinite;
      `;
      
      // Add to toolbar
      const toolbar = document.querySelector('.toolbar-right');
      if (toolbar) {
        toolbar.insertBefore(indicator, toolbar.firstChild);
      }
      
      // Add spin animation
      if (!document.getElementById('markajiap-refresh-style')) {
        const style = document.createElement('style');
        style.id = 'markajiap-refresh-style';
        style.textContent = `
          @keyframes spin { from { transform: rotate(0deg); } to { transform: rotate(360deg); } }
          @keyframes flash { 0%, 100% { opacity: 0.7; } 50% { opacity: 1; color: #D4A634; } }
        `;
        document.head.appendChild(style);
      }
    } else if (!show && indicator) {
      indicator.remove();
    }
  }

  // Flash indicator when content refreshes
  flashRefreshIndicator() {
    const indicator = document.getElementById('markajiap-refresh-indicator');
    if (indicator) {
      indicator.style.animation = 'flash 0.5s ease';
      setTimeout(() => {
        indicator.style.animation = 'spin 2s linear infinite';
      }, 500);
    }
  }

  createRenderedView(markdown, options) {
    console.log('🏗️ MarkaJiap: createRenderedView() called');
    
    document.body.innerHTML = '';
    document.body.style.margin = '0';
    document.body.style.padding = '0';
    
    // Resolve 'auto' theme to actual light/dark based on system preference
    const effectiveTheme = this.getEffectiveTheme(options.theme);
    document.documentElement.setAttribute('data-theme', effectiveTheme);
    
    // Setup listener for system theme changes
    this.setupThemeListener();

    this.injectStyles(options);
    console.log('🏗️ MarkaJiap: Styles injected');

    const container = document.createElement('div');
    container.id = 'markajiap-container';
    container.innerHTML = `
      <div id="markajiap-sidebar" class="sidebar">
        <div class="sidebar-header">
          <h3>📑 Contents</h3>
          <button id="toggle-sidebar" title="Close">×</button>
        </div>
        <nav id="markajiap-toc"></nav>
      </div>
      <main id="markajiap-main">
        <div class="toolbar">
          <div class="toolbar-left">
            <span class="logo">🔮 <strong>Marka</strong>Jiap</span>
            <span class="file-name">${this.getFilename()}</span>
          </div>
          <div class="toolbar-right">
            <button id="btn-export" title="Export">📄</button>
            <button id="btn-refresh" title="Toggle Auto-Refresh">🔄</button>
            <button id="btn-theme" title="Toggle Theme">🌓</button>
            <button id="btn-toc" title="Toggle TOC">📑</button>
            <button id="btn-print" title="Print">🖨️</button>
          </div>
        </div>
        <div class="progress-bar"><div class="progress-fill"></div></div>
        <article id="markajiap-content" class="markdown-body"></article>
        <button id="scroll-top" title="Top">↑</button>
      </main>
    `;

    document.body.appendChild(container);
    console.log('🏗️ MarkaJiap: Container appended to body');
    console.log('🏗️ MarkaJiap: Calling renderMarkdown with markdown length:', markdown.length);

    this.renderMarkdown(markdown, options);
    console.log('🏗️ MarkaJiap: renderMarkdown completed');
    
    this.setupInteractions(options);
    console.log('🏗️ MarkaJiap: setupInteractions completed');

    const h1 = document.querySelector('#markajiap-content h1, #markajiap-content h2');
    if (h1) document.title = h1.textContent + ' | MarkaJiap';
    
    console.log('🏗️ MarkaJiap: createRenderedView completed');
  }

  getFilename() {
    return window.location.pathname.split('/').pop() || 'Document';
  }

  // Map font option to CSS font-family stack
  getFontStack(fontOption) {
    const fontStacks = {
      'system': '-apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Helvetica, Arial, sans-serif',
      'ibm-plex': '"IBM Plex Sans", -apple-system, sans-serif',
      'inter': '"Inter", -apple-system, sans-serif',
      'roboto': '"Roboto", -apple-system, sans-serif',
      'source-sans': '"Source Sans Pro", -apple-system, sans-serif',
      'georgia': 'Georgia, "Times New Roman", serif',
      'merriweather': '"Merriweather", Georgia, serif'
    };
    return fontStacks[fontOption] || fontStacks['ibm-plex'];
  }

  injectStyles(options) {
    const style = document.createElement('style');
    style.id = 'markajiap-styles';
    style.textContent = `
      @import url('https://fonts.googleapis.com/css2?family=IBM+Plex+Sans:wght@400;500;600;700&family=IBM+Plex+Mono:wght@400;500&display=swap');

      :root {
        --bg-primary: #FFFFFF;
        --bg-secondary: #FAF8FC;
        --bg-tertiary: #EDE9F0;
        --text-primary: #1A1625;
        --text-secondary: #6B6B7B;
        --accent: #7B2D8E;
        --accent-gold: #D4A634;
        --border-color: #EDE9F0;
        --code-bg: #F6F3F8;
        --sidebar-width: 280px;
      }

      [data-theme="dark"] {
        --bg-primary: #0D0A14;
        --bg-secondary: #16121F;
        --bg-tertiary: #1F1A2E;
        --text-primary: #F0EDF5;
        --text-secondary: #A09AAD;
        --border-color: #2A243A;
        --code-bg: #1A1528;
      }

      * { box-sizing: border-box; margin: 0; padding: 0; }

      body {
        font-family: ${this.getFontStack(options.fontFamily)};
        font-size: ${options.fontSize}px;
        line-height: ${options.lineHeight};
        color: var(--text-primary);
        background: var(--bg-primary);
      }

      #markajiap-container { display: flex; min-height: 100vh; }

      .sidebar {
        width: var(--sidebar-width);
        background: var(--bg-secondary);
        border-right: 1px solid var(--border-color);
        position: fixed;
        top: 0; left: 0;
        height: 100vh;
        overflow-y: auto;
        z-index: 100;
        transition: transform 0.3s;
      }

      .sidebar.collapsed { transform: translateX(-100%); }

      .sidebar-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        padding: 16px 20px;
        border-bottom: 1px solid var(--border-color);
        position: sticky;
        top: 0;
        background: var(--bg-secondary);
      }

      .sidebar-header h3 { font-size: 14px; font-weight: 600; }

      .sidebar-header button {
        background: none;
        border: none;
        font-size: 20px;
        cursor: pointer;
        color: var(--text-secondary);
        padding: 4px 8px;
        border-radius: 4px;
      }

      .sidebar-header button:hover { background: var(--bg-tertiary); }

      #markajiap-toc { padding: 16px; }

      #markajiap-toc a {
        display: block;
        padding: 8px 12px;
        color: var(--text-secondary);
        text-decoration: none;
        font-size: 13px;
        border-radius: 6px;
        margin-bottom: 2px;
        transition: all 0.2s;
      }

      #markajiap-toc a:hover {
        background: var(--bg-tertiary);
        color: var(--text-primary);
      }

      #markajiap-toc a.active {
        background: var(--accent);
        color: white;
      }

      #markajiap-toc a.level-2 { padding-left: 24px; }
      #markajiap-toc a.level-3 { padding-left: 36px; font-size: 12px; }

      #markajiap-main {
        flex: 1;
        margin-left: var(--sidebar-width);
        transition: margin-left 0.3s;
      }

      .sidebar.collapsed + #markajiap-main { margin-left: 0; }

      .toolbar {
        display: flex;
        justify-content: space-between;
        align-items: center;
        padding: 12px 24px;
        background: var(--bg-secondary);
        border-bottom: 1px solid var(--border-color);
        position: sticky;
        top: 0;
        z-index: 50;
      }

      .toolbar-left {
        display: flex;
        align-items: center;
        gap: 16px;
      }

      .toolbar-left .logo { font-size: 16px; }
      .toolbar-left .file-name { font-size: 13px; color: var(--text-secondary); }

      .toolbar-right { display: flex; gap: 8px; }

      .toolbar-right button {
        background: var(--bg-tertiary);
        border: none;
        padding: 8px 12px;
        border-radius: 6px;
        cursor: pointer;
        font-size: 16px;
        transition: all 0.2s;
      }

      .toolbar-right button:hover {
        background: var(--accent);
        transform: translateY(-1px);
      }

      .progress-bar { height: 3px; background: var(--border-color); }

      .progress-fill {
        height: 100%;
        width: 0%;
        background: linear-gradient(90deg, var(--accent), var(--accent-gold));
        transition: width 0.1s;
      }

      #markajiap-content {
        max-width: 900px;
        margin: 0 auto;
        padding: 40px 24px 100px;
      }

      .markdown-body h1, .markdown-body h2, .markdown-body h3,
      .markdown-body h4, .markdown-body h5, .markdown-body h6 {
        margin-top: 1.5em;
        margin-bottom: 0.5em;
        font-weight: 600;
        line-height: 1.3;
      }

      .markdown-body h1 { font-size: 2em; border-bottom: 2px solid var(--accent); padding-bottom: 0.3em; }
      .markdown-body h2 { font-size: 1.5em; border-bottom: 1px solid var(--border-color); padding-bottom: 0.3em; }
      .markdown-body h3 { font-size: 1.25em; }

      .markdown-body p { margin: 1em 0; }

      .markdown-body a { color: var(--accent); text-decoration: none; }
      .markdown-body a:hover { text-decoration: underline; }

      .markdown-body code {
        font-family: 'IBM Plex Mono', monospace;
        background: var(--code-bg);
        padding: 2px 6px;
        border-radius: 4px;
        font-size: 0.9em;
      }

      .markdown-body pre {
        background: var(--code-bg);
        padding: 16px;
        border-radius: 8px;
        overflow-x: auto;
        margin: 1em 0;
        position: relative;
      }

      .markdown-body pre code { background: none; padding: 0; }

      .markdown-body pre::before {
        content: attr(data-lang);
        position: absolute;
        top: 8px;
        right: 12px;
        font-size: 11px;
        color: var(--text-secondary);
        text-transform: uppercase;
      }

      .markdown-body blockquote {
        margin: 1em 0;
        padding: 12px 20px;
        border-left: 4px solid var(--accent);
        background: var(--bg-secondary);
        border-radius: 0 8px 8px 0;
      }

      .markdown-body table { width: 100%; border-collapse: collapse; margin: 1em 0; }

      .markdown-body th, .markdown-body td {
        padding: 12px;
        border: 1px solid var(--border-color);
        text-align: left;
      }

      .markdown-body th { background: var(--bg-secondary); font-weight: 600; }

      .markdown-body tr:hover { background: var(--bg-secondary); }

      .markdown-body img { max-width: 100%; border-radius: 8px; }

      .markdown-body ul, .markdown-body ol { padding-left: 2em; margin: 1em 0; }

      .markdown-body li { margin: 0.5em 0; }

      .markdown-body hr { border: none; border-top: 1px solid var(--border-color); margin: 2em 0; }

      .markdown-body .task-list-item { list-style: none; margin-left: -1.5em; }

      .markdown-body .task-list-item input { margin-right: 8px; accent-color: var(--accent); }

      .mermaid {
        background: var(--bg-secondary);
        padding: 24px;
        border-radius: 12px;
        text-align: center;
        margin: 1.5em 0;
        border: 1px solid var(--border-color);
        overflow-x: auto;
      }

      .mermaid svg {
        max-width: 100%;
        height: auto;
      }

      /* Mermaid text colors for our theme */
      .mermaid .nodeLabel,
      .mermaid .label,
      .mermaid text {
        font-family: 'IBM Plex Sans', -apple-system, sans-serif !important;
      }

      .mermaid .node rect,
      .mermaid .node circle,
      .mermaid .node polygon {
        stroke-width: 2px;
      }

      /* Loading state */
      .mermaid:not(:has(svg)) {
        min-height: 100px;
        display: flex;
        align-items: center;
        justify-content: center;
        color: var(--text-secondary);
        font-family: 'IBM Plex Mono', monospace;
        font-size: 13px;
        white-space: pre-wrap;
        text-align: left;
      }

      /* Mermaid wrapper & controls */
      .mermaid-wrapper {
        position: relative;
        margin: 1em 0;
      }

      .mermaid-controls {
        position: absolute;
        top: 8px;
        right: 8px;
        display: flex;
        gap: 4px;
        opacity: 0;
        transition: opacity 0.2s;
      }

      .mermaid-wrapper:hover .mermaid-controls {
        opacity: 1;
      }

      .mermaid-btn {
        background: var(--bg-tertiary);
        border: 1px solid var(--border-color);
        border-radius: 4px;
        padding: 6px 8px;
        cursor: pointer;
        display: flex;
        align-items: center;
        justify-content: center;
        color: var(--text-primary);
        transition: all 0.2s;
      }

      .mermaid-btn:hover {
        background: var(--accent);
        color: white;
      }

      /* Download Dialog */
      #download-dialog {
        position: fixed;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        z-index: 10000;
        display: flex;
        align-items: center;
        justify-content: center;
      }

      .dialog-overlay {
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: rgba(0,0,0,0.7);
      }

      .dialog-content {
        position: relative;
        background: var(--bg-secondary);
        border-radius: 12px;
        width: 400px;
        max-width: 90%;
        box-shadow: 0 20px 60px rgba(0,0,0,0.5);
      }

      .dialog-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        padding: 16px 20px;
        border-bottom: 1px solid var(--border-color);
      }

      .dialog-header h3 {
        margin: 0;
        font-size: 16px;
        font-weight: 600;
      }

      .dialog-close {
        background: none;
        border: none;
        font-size: 24px;
        cursor: pointer;
        color: var(--text-secondary);
        padding: 0;
        line-height: 1;
      }

      .dialog-close:hover {
        color: var(--text-primary);
      }

      .dialog-body {
        padding: 20px;
      }

      .dialog-body label {
        display: block;
        font-size: 13px;
        font-weight: 500;
        color: var(--text-secondary);
        margin-bottom: 8px;
        margin-top: 16px;
      }

      .dialog-body label:first-child {
        margin-top: 0;
      }

      .dialog-body select {
        width: 100%;
        padding: 12px;
        border: 1px solid var(--border-color);
        border-radius: 8px;
        background: var(--bg-tertiary);
        color: var(--text-primary);
        font-size: 14px;
        cursor: pointer;
      }

      .dialog-footer {
        display: flex;
        justify-content: flex-end;
        gap: 12px;
        padding: 16px 20px;
        border-top: 1px solid var(--border-color);
      }

      .dialog-footer button {
        padding: 10px 20px;
        border-radius: 8px;
        font-size: 14px;
        font-weight: 500;
        cursor: pointer;
        transition: all 0.2s;
      }

      .btn-cancel {
        background: var(--bg-tertiary);
        border: 1px solid var(--border-color);
        color: var(--text-primary);
      }

      .btn-cancel:hover {
        background: var(--border-color);
      }

      .btn-download {
        background: var(--accent);
        border: none;
        color: white;
      }

      .btn-download:hover {
        background: #6a2579;
        transform: translateY(-1px);
      }

      /* Media Viewer (fullscreen) */
      #media-viewer {
        position: fixed;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: rgba(13, 10, 20, 0.95);
        z-index: 10001;
        display: flex;
        flex-direction: column;
      }

      .viewer-controls {
        position: absolute;
        top: 16px;
        right: 16px;
        display: flex;
        gap: 8px;
        z-index: 10;
      }

      .viewer-controls button {
        width: 40px;
        height: 40px;
        background: var(--bg-tertiary);
        border: 1px solid var(--border-color);
        border-radius: 8px;
        font-size: 20px;
        cursor: pointer;
        color: var(--text-primary);
        display: flex;
        align-items: center;
        justify-content: center;
        transition: all 0.2s;
      }

      .viewer-controls button:hover {
        background: var(--accent);
        color: white;
        transform: scale(1.05);
      }

      .viewer-content {
        flex: 1;
        display: flex;
        align-items: center;
        justify-content: center;
        overflow: hidden;
        cursor: grab;
      }

      .viewer-content:active {
        cursor: grabbing;
      }

      .viewer-transform {
        transition: transform 0.1s ease-out;
      }

      .viewer-transform img {
        max-width: none;
        max-height: none;
        user-select: none;
        -webkit-user-drag: none;
      }

      .mermaid-clone {
        background: var(--bg-secondary);
        padding: 40px;
        border-radius: 12px;
      }

      .mermaid-clone svg {
        max-width: none !important;
        max-height: none !important;
      }

      #scroll-top {
        position: fixed;
        bottom: 80px;
        right: 20px;
        width: 44px;
        height: 44px;
        background: var(--accent);
        color: white;
        border: none;
        border-radius: 50%;
        font-size: 20px;
        cursor: pointer;
        opacity: 0;
        transition: all 0.3s;
        z-index: 100;
      }

      #scroll-top.visible { opacity: 1; }

      #scroll-top:hover {
        transform: translateY(-2px);
        box-shadow: 0 4px 12px rgba(123, 45, 142, 0.4);
      }

      #markajiap-btn { bottom: 20px !important; }

      @media (max-width: 768px) {
        .sidebar { transform: translateX(-100%); }
        .sidebar.open { transform: translateX(0); }
        #markajiap-main { margin-left: 0 !important; }
        #markajiap-content { padding: 20px 16px 100px; }
      }

      ${options.customCss}
    `;
    document.head.appendChild(style);
  }

  renderMarkdown(markdown, options) {
    console.log('📝 MarkaJiap: renderMarkdown called');
    
    const contentEl = document.getElementById('markajiap-content');
    
    // Store raw markdown in data attribute for the external renderer script
    contentEl.setAttribute('data-raw-markdown', markdown);
    contentEl.innerHTML = '<p>📝 Loading markdown renderer...</p>';
    
    // Store options for later use
    this._renderOptions = options;
    
    // Listen for render complete event
    const handleRenderComplete = () => {
      console.log('📝 MarkaJiap: Render complete event received');
      window.removeEventListener('markajiap-render-complete', handleRenderComplete);
      
      if (this._renderOptions.showToc) {
        this.generateToc();
      }

      // Setup image zoom
      this.setupImageZoom();

      // Render mermaid diagrams
      console.log('📝 MarkaJiap: Calling renderMermaid...');
      this.renderMermaid();

      // Load and apply syntax highlighting
      this.loadHighlightJS(this._renderOptions.theme);

      // Setup mermaid controls (after render)
      setTimeout(() => this.setupMermaidControls(), 1500);
      
      console.log('📝 MarkaJiap: renderMarkdown completed');
    };
    window.addEventListener('markajiap-render-complete', handleRenderComplete);
    
    // Load marked.js first
    const markedScript = document.createElement('script');
    markedScript.src = chrome.runtime.getURL('lib/marked.min.js');
    markedScript.onload = () => {
      console.log('📝 MarkaJiap: marked.js loaded');
      
      // Then load the renderer script (runs in page context)
      const rendererScript = document.createElement('script');
      rendererScript.src = chrome.runtime.getURL('lib/markdownRenderer.js');
      rendererScript.onerror = (e) => {
        console.error('❌ MarkaJiap: Failed to load markdownRenderer.js', e);
        this.fallbackRender(markdown, options);
      };
      document.body.appendChild(rendererScript);
    };
    
    markedScript.onerror = (e) => {
      console.error('❌ MarkaJiap: Failed to load marked.js', e);
      this.fallbackRender(markdown, options);
    };
    
    document.head.appendChild(markedScript);
  }
  
  // Fallback render using simple parser
  fallbackRender(markdown, options) {
    console.log('📝 MarkaJiap: Using fallback parser');
    const contentEl = document.getElementById('markajiap-content');
    contentEl.removeAttribute('data-raw-markdown');
    let html = this.markdownToHtml(markdown);
    contentEl.innerHTML = html;
    
    if (options.showToc) {
      this.generateToc();
    }
    this.setupImageZoom();
    this.renderMermaid();
    this.loadHighlightJS(options.theme, options.codeTheme);
    setTimeout(() => this.setupMermaidControls(), 1500);
  }

  // Load highlight.js dynamically and highlight all code blocks
  loadHighlightJS(theme, codeTheme = 'markajiap') {
    console.log('📝 MarkaJiap: Loading highlight.js...');
    
    // Inject syntax highlighting CSS
    this.injectSyntaxHighlightingCSS(theme, codeTheme);
    
    // Load highlight.js from local lib (CSP compliant)
    const hljsScript = document.createElement('script');
    hljsScript.src = chrome.runtime.getURL('lib/highlight.min.js');
    hljsScript.onload = () => {
      console.log('📝 MarkaJiap: highlight.js loaded');
      
      // Load external highlighter script (CSP compliant - no inline scripts)
      const highlighterScript = document.createElement('script');
      highlighterScript.src = chrome.runtime.getURL('lib/highlighter.js');
      highlighterScript.onerror = (e) => {
        console.error('❌ MarkaJiap: Failed to load highlighter.js', e);
      };
      document.body.appendChild(highlighterScript);
    };
    hljsScript.onerror = (e) => {
      console.error('❌ MarkaJiap: Failed to load highlight.js', e);
    };
    document.head.appendChild(hljsScript);
  }

  // Inject syntax highlighting CSS based on theme and code theme
  injectSyntaxHighlightingCSS(theme, codeTheme = 'markajiap') {
    // Remove existing styles
    document.getElementById('hljs-inline-css')?.remove();
    document.getElementById('hljs-external-css')?.remove();

    const isDark = theme === 'dark';
    
    // Map code theme to CSS file
    const themeFiles = {
      'markajiap': isDark ? 'hljs-markajiap-dark.css' : 'hljs-markajiap.css',
      'github': isDark ? 'hljs-github-dark.css' : 'hljs-github.css',
      'atom-one-dark': 'atom-one-dark.min.css',
      'atom-one-light': 'atom-one-light.min.css',
      'monokai': 'monokai.min.css',
      'dracula': 'dracula.min.css',
      'nord': 'nord.min.css',
      'vs2015': 'vs2015.min.css'
    };
    
    // Use MarkaJiap theme with inline CSS for default
    if (codeTheme === 'markajiap') {
      const style = document.createElement('style');
      style.id = 'hljs-inline-css';
      
      if (isDark) {
        style.textContent = `
          .hljs { color: #F0EDF5 !important; background: transparent !important; }
          .hljs-keyword, .hljs-type, .hljs-built_in, .hljs-doctag, .hljs-variable.language_ { color: #C77DFF !important; font-weight: 600; }
          .hljs-title, .hljs-title.class_, .hljs-title.function_ { color: #E0AAFF !important; font-weight: 600; }
          .hljs-string, .hljs-meta .hljs-string, .hljs-regexp { color: #E8C864 !important; }
          .hljs-number, .hljs-literal { color: #79C0FF !important; }
          .hljs-attr { color: #FF92DF !important; }
          .hljs-attribute, .hljs-variable { color: #58D1F0 !important; }
          .hljs-symbol, .hljs-bullet { color: #FFD700 !important; }
          .hljs-comment, .hljs-code, .hljs-formula { color: #8B949E !important; font-style: italic; }
          .hljs-name, .hljs-selector-tag, .hljs-quote { color: #7EE787 !important; }
          .hljs-operator, .hljs-meta { color: #FF79C6 !important; }
          .hljs-property { color: #79C0FF !important; }
          .hljs-params { color: #FFA657 !important; }
          .hljs-punctuation { color: #A09AAD !important; }
          .hljs-section { color: #C77DFF !important; font-weight: 700; }
          .hljs-subst { color: #F0EDF5 !important; }
        `;
      } else {
        style.textContent = `
          .hljs { color: #1A1625 !important; background: transparent !important; }
          .hljs-keyword, .hljs-type, .hljs-built_in, .hljs-doctag, .hljs-variable.language_ { color: #7B2D8E !important; font-weight: 600; }
          .hljs-title, .hljs-title.class_, .hljs-title.function_ { color: #5A1F6B !important; font-weight: 600; }
          .hljs-string, .hljs-meta .hljs-string, .hljs-regexp { color: #986801 !important; }
          .hljs-number, .hljs-literal { color: #005CC5 !important; }
          .hljs-attr { color: #D63384 !important; }
          .hljs-attribute, .hljs-variable { color: #0D7377 !important; }
          .hljs-symbol, .hljs-bullet { color: #D4A634 !important; }
          .hljs-comment, .hljs-code, .hljs-formula { color: #6A737D !important; font-style: italic; }
          .hljs-name, .hljs-selector-tag, .hljs-quote { color: #22863A !important; }
          .hljs-operator, .hljs-meta { color: #D73A49 !important; }
          .hljs-property { color: #005CC5 !important; }
          .hljs-params { color: #E36209 !important; }
          .hljs-punctuation { color: #6B6B7B !important; }
          .hljs-section { color: #7B2D8E !important; font-weight: 700; }
          .hljs-subst { color: #1A1625 !important; }
        `;
      }
      document.head.appendChild(style);
    } else {
      // Load external CSS file for other themes
      const cssFile = themeFiles[codeTheme] || themeFiles['markajiap'];
      const link = document.createElement('link');
      link.id = 'hljs-external-css';
      link.rel = 'stylesheet';
      link.href = chrome.runtime.getURL('lib/' + cssFile);
      document.head.appendChild(link);
      console.log('🎨 MarkaJiap: Loaded code theme:', codeTheme, '→', cssFile);
    }
  }

  renderMermaid() {
    console.log('🔮 MarkaJiap: renderMermaid() called');
    
    const mermaidEls = document.querySelectorAll('.mermaid');
    console.log('🔮 MarkaJiap: querySelectorAll found', mermaidEls.length, 'elements');
    
    if (mermaidEls.length === 0) {
      console.log('🔮 MarkaJiap: No mermaid elements found, exiting');
      return;
    }

    // Store theme settings in DOM for the render script to read
    const isDark = document.documentElement.getAttribute('data-theme') === 'dark';
    const configEl = document.createElement('div');
    configEl.id = 'markajiap-mermaid-config';
    configEl.dataset.dark = isDark.toString();
    configEl.style.display = 'none';
    document.body.appendChild(configEl);
    
    console.log('🔮 MarkaJiap: Config element added, dark:', isDark);

    // Get script URLs from extension
    const mermaidUrl = chrome.runtime.getURL('lib/mermaid.min.js');
    const renderUrl = chrome.runtime.getURL('js/mermaid-render.js');
    
    console.log('🔮 MarkaJiap: Mermaid URL:', mermaidUrl);
    console.log('🔮 MarkaJiap: Render URL:', renderUrl);

    // Load mermaid library first
    const mermaidScript = document.createElement('script');
    mermaidScript.src = mermaidUrl;
    
    mermaidScript.onload = () => {
      console.log('✅ MarkaJiap: Mermaid library loaded');
      
      // Then load the render script
      const renderScript = document.createElement('script');
      renderScript.src = renderUrl;
      
      renderScript.onload = () => {
        console.log('✅ MarkaJiap: Render script loaded');
        setTimeout(() => this.setupMermaidControls(), 2000);
      };
      
      renderScript.onerror = (e) => {
        console.error('❌ MarkaJiap: Failed to load render script:', e);
        this.showMermaidError();
      };
      
      document.head.appendChild(renderScript);
    };
    
    mermaidScript.onerror = (e) => {
      console.error('❌ MarkaJiap: Failed to load mermaid library:', e);
      this.showMermaidError();
    };
    
    document.head.appendChild(mermaidScript);
  }

  showMermaidError() {
    console.log('⚠️ MarkaJiap: showMermaidError called');
    document.querySelectorAll('.mermaid').forEach(el => {
      const code = el.textContent;
      el.innerHTML = `
        <div style="background: #1F1A2E; border: 1px solid #7B2D8E; border-radius: 8px; padding: 16px; text-align: left;">
          <div style="color: #D4A634; font-size: 12px; margin-bottom: 8px;">⚠️ Mermaid diagram (requires network)</div>
          <pre style="color: #A09AAD; font-family: 'IBM Plex Mono', monospace; font-size: 12px; margin: 0; white-space: pre-wrap;">${this.escapeHtml(code)}</pre>
        </div>
      `;
    });
  }

  initMermaid() {
    const isDark = document.documentElement.getAttribute('data-theme') === 'dark';
    
    console.log('🎨 MarkaJiap: initMermaid() called');
    console.log('🎨 MarkaJiap: Dark mode:', isDark);
    console.log('🎨 MarkaJiap: typeof mermaid:', typeof mermaid);
    console.log('🎨 MarkaJiap: typeof window.mermaid:', typeof window.mermaid);

    try {
      // Use window.mermaid since the UMD build exports there
      const mermaidLib = window.mermaid;
      
      if (!mermaidLib) {
        console.error('❌ MarkaJiap: window.mermaid is not defined!');
        this.showMermaidError();
        return;
      }
      
      console.log('🎨 MarkaJiap: mermaidLib found:', mermaidLib);
      console.log('🎨 MarkaJiap: mermaid.initialize exists:', typeof mermaidLib.initialize);
      console.log('🎨 MarkaJiap: mermaid.render exists:', typeof mermaidLib.render);
      
      // MarkaJiap custom theme with lucky colors
      mermaidLib.initialize({
        startOnLoad: false,
        theme: 'base',
        securityLevel: 'loose',
        themeVariables: {
          // Primary colors - Purple (โชคลาภ)
          primaryColor: '#7B2D8E',
          primaryTextColor: '#FFFFFF',
          primaryBorderColor: '#5E2270',
          
          // Secondary colors - Gold (บารมี)
          secondaryColor: '#D4A634',
          secondaryTextColor: '#1A1625',
          secondaryBorderColor: '#B8922D',
          
          // Tertiary
          tertiaryColor: '#9B3DAE',
          tertiaryTextColor: '#FFFFFF',
          tertiaryBorderColor: '#7B2D8E',
          
          // Background
          background: isDark ? '#16121F' : '#FAF8FC',
          mainBkg: isDark ? '#1F1A2E' : '#FFFFFF',
          
          // Node colors
          nodeBkg: isDark ? '#2A243A' : '#FAF8FC',
          nodeBorder: '#7B2D8E',
          
          // Text
          textColor: isDark ? '#F0EDF5' : '#1A1625',
          
          // Lines
          lineColor: '#7B2D8E',
          
          // Flowchart
          clusterBkg: isDark ? '#1F1A2E' : '#EDE9F0',
          clusterBorder: '#7B2D8E',
          
          // Sequence diagram
          actorBkg: '#7B2D8E',
          actorBorder: '#5E2270',
          actorTextColor: '#FFFFFF',
          actorLineColor: '#7B2D8E',
          signalColor: '#D4A634',
          signalTextColor: isDark ? '#F0EDF5' : '#1A1625',
          
          // Notes
          noteBkgColor: '#D4A634',
          noteTextColor: '#1A1625',
          noteBorderColor: '#B8922D',
          
          // Labels
          labelColor: isDark ? '#F0EDF5' : '#1A1625',
          labelTextColor: isDark ? '#F0EDF5' : '#1A1625',
          
          // Edges
          edgeLabelBackground: isDark ? '#1F1A2E' : '#FFFFFF',
          
          // Pie chart
          pie1: '#7B2D8E',
          pie2: '#D4A634',
          pie3: '#9B3DAE',
          pie4: '#F4D675',
          pie5: '#5E2270',
          pie6: '#B8922D',
          
          // Fonts
          fontFamily: '"IBM Plex Sans", -apple-system, sans-serif',
          fontSize: '14px'
        },
        flowchart: {
          useMaxWidth: true,
          htmlLabels: true,
          curve: 'basis'
        }
      });
      
      console.log('🎨 MarkaJiap: mermaid.initialize() completed');

      // Render each diagram
      this.renderMermaidDiagrams();
      
    } catch (err) {
      console.error('❌ MarkaJiap: Mermaid init error:', err);
      console.error('❌ MarkaJiap: Error stack:', err.stack);
      this.showMermaidError();
    }
  }

  async renderMermaidDiagrams() {
    console.log('🖼️ MarkaJiap: renderMermaidDiagrams() called');
    
    const mermaidLib = window.mermaid;
    
    if (!mermaidLib) {
      console.error('❌ MarkaJiap: window.mermaid not available in renderMermaidDiagrams');
      return;
    }
    
    const mermaidEls = document.querySelectorAll('.mermaid');
    
    console.log('🖼️ MarkaJiap: Found', mermaidEls.length, 'mermaid elements to render');
    
    for (let i = 0; i < mermaidEls.length; i++) {
      const el = mermaidEls[i];
      const code = el.textContent.trim();
      
      console.log(`🖼️ MarkaJiap: Processing diagram ${i}:`, code.substring(0, 50) + '...');
      
      if (!code) {
        console.log(`🖼️ MarkaJiap: Diagram ${i} has no code, skipping`);
        continue;
      }
      
      if (el.querySelector('svg')) {
        console.log(`🖼️ MarkaJiap: Diagram ${i} already has SVG, skipping`);
        continue;
      }
      
      const id = `mermaid-diagram-${i}-${Date.now()}`;
      console.log(`🖼️ MarkaJiap: Rendering diagram ${i} with id: ${id}`);
      
      try {
        console.log(`🖼️ MarkaJiap: Calling mermaid.render for diagram ${i}...`);
        
        // Use mermaid.render (returns Promise in v10+)
        const result = await mermaidLib.render(id, code);
        console.log(`🖼️ MarkaJiap: mermaid.render returned for diagram ${i}:`, result);
        
        const { svg } = result;
        console.log(`🖼️ MarkaJiap: SVG length for diagram ${i}:`, svg ? svg.length : 'null');
        
        el.innerHTML = svg;
        console.log(`✅ MarkaJiap: Diagram ${i} rendered successfully`);
        
      } catch (err) {
        console.error(`❌ MarkaJiap: Diagram ${i} error:`, err);
        console.error(`❌ MarkaJiap: Diagram ${i} code was:`, code);
        el.innerHTML = `
          <div style="background: #1F1A2E; border: 1px solid #E63946; border-radius: 8px; padding: 16px; text-align: left;">
            <div style="color: #E63946; font-size: 12px; margin-bottom: 8px;">❌ Syntax Error</div>
            <pre style="color: #A09AAD; font-family: 'IBM Plex Mono', monospace; font-size: 12px; margin: 0; white-space: pre-wrap;">${err.message || 'Unknown error'}</pre>
          </div>
        `;
      }
    }
    
    console.log('🖼️ MarkaJiap: renderMermaidDiagrams() completed');
  }

  setupImageZoom() {
    // Load lightbox script
    const loadLightbox = () => {
      return new Promise((resolve) => {
        if (window.MarkaJiapLightbox) {
          resolve();
          return;
        }
        
        const script = document.createElement('script');
        script.src = chrome.runtime.getURL('lib/lightbox.js');
        script.onload = resolve;
        script.onerror = () => {
          console.error('📷 MarkaJiap: Failed to load lightbox');
          resolve(); // Continue anyway
        };
        document.body.appendChild(script);
      });
    };
    
    // Collect all images
    const allImages = Array.from(document.querySelectorAll('#markajiap-content img'));
    
    allImages.forEach((img, index) => {
      img.style.cursor = 'zoom-in';
      
      img.addEventListener('click', async () => {
        await loadLightbox();
        
        if (window.MarkaJiapLightbox) {
          // Build image array for gallery
          const images = allImages.map(i => ({
            src: i.src,
            alt: i.alt || i.title || ''
          }));
          
          window.MarkaJiapLightbox.open(images, index);
        } else {
          // Fallback to old media viewer
          this.openMediaViewer(img.src, 'image');
        }
      });
    });
    
    console.log('📷 MarkaJiap: Image zoom setup for', allImages.length, 'images');
  }

  setupMermaidControls() {
    document.querySelectorAll('.mermaid').forEach((mermaidEl, index) => {
      // Wrap in container
      const wrapper = document.createElement('div');
      wrapper.className = 'mermaid-wrapper';
      mermaidEl.parentNode.insertBefore(wrapper, mermaidEl);
      wrapper.appendChild(mermaidEl);

      // Add controls
      const controls = document.createElement('div');
      controls.className = 'mermaid-controls';
      controls.innerHTML = `
        <button class="mermaid-btn" data-action="download" title="Download">
          <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
            <path d="M21 15v4a2 2 0 01-2 2H5a2 2 0 01-2-2v-4"/>
            <polyline points="7,10 12,15 17,10"/>
            <line x1="12" y1="15" x2="12" y2="3"/>
          </svg>
        </button>
        <button class="mermaid-btn" data-action="fullscreen" title="Fullscreen">
          <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
            <path d="M8 3H5a2 2 0 00-2 2v3m18 0V5a2 2 0 00-2-2h-3m0 18h3a2 2 0 002-2v-3M3 16v3a2 2 0 002 2h3"/>
          </svg>
        </button>
      `;
      wrapper.appendChild(controls);

      // Event handlers
      controls.querySelector('[data-action="download"]').addEventListener('click', () => {
        this.showDownloadDialog(mermaidEl, index);
      });

      controls.querySelector('[data-action="fullscreen"]').addEventListener('click', () => {
        this.openMediaViewer(mermaidEl, 'mermaid');
      });
    });
  }

  showDownloadDialog(mermaidEl, index) {
    // Remove existing dialog
    document.getElementById('download-dialog')?.remove();

    const dialog = document.createElement('div');
    dialog.id = 'download-dialog';
    dialog.innerHTML = `
      <div class="dialog-overlay"></div>
      <div class="dialog-content">
        <div class="dialog-header">
          <h3>Download Diagram</h3>
          <button class="dialog-close">×</button>
        </div>
        <div class="dialog-body">
          <label>Format</label>
          <select id="download-format">
            <option value="png">PNG</option>
            <option value="svg">SVG</option>
          </select>
          <label>Scale</label>
          <select id="download-scale">
            <option value="1">1x (Standard)</option>
            <option value="2" selected>2x (High Quality)</option>
            <option value="3">3x (Ultra)</option>
          </select>
        </div>
        <div class="dialog-footer">
          <button class="btn-cancel">Cancel</button>
          <button class="btn-download">Download</button>
        </div>
      </div>
    `;

    document.body.appendChild(dialog);

    // Close handlers
    dialog.querySelector('.dialog-overlay').addEventListener('click', () => dialog.remove());
    dialog.querySelector('.dialog-close').addEventListener('click', () => dialog.remove());
    dialog.querySelector('.btn-cancel').addEventListener('click', () => dialog.remove());

    // Download handler
    dialog.querySelector('.btn-download').addEventListener('click', () => {
      const format = document.getElementById('download-format').value;
      const scale = parseInt(document.getElementById('download-scale').value);
      this.downloadDiagram(mermaidEl, format, scale, index);
      dialog.remove();
    });
  }

  downloadDiagram(mermaidEl, format, scale, index) {
    const svg = mermaidEl.querySelector('svg');
    if (!svg) {
      // If no SVG yet, try to get the raw content
      alert('Diagram not rendered yet');
      return;
    }

    if (format === 'svg') {
      // Download as SVG
      const svgData = new XMLSerializer().serializeToString(svg);
      const blob = new Blob([svgData], { type: 'image/svg+xml' });
      this.downloadBlob(blob, `diagram-${index + 1}.svg`);
    } else {
      // Download as PNG
      const svgData = new XMLSerializer().serializeToString(svg);
      const canvas = document.createElement('canvas');
      const ctx = canvas.getContext('2d');
      const img = new Image();

      img.onload = () => {
        canvas.width = img.width * scale;
        canvas.height = img.height * scale;
        ctx.scale(scale, scale);
        ctx.fillStyle = '#1a1a2e';
        ctx.fillRect(0, 0, canvas.width, canvas.height);
        ctx.drawImage(img, 0, 0);

        canvas.toBlob((blob) => {
          this.downloadBlob(blob, `diagram-${index + 1}.png`);
        }, 'image/png');
      };

      img.src = 'data:image/svg+xml;base64,' + btoa(unescape(encodeURIComponent(svgData)));
    }
  }

  downloadBlob(blob, filename) {
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = filename;
    a.click();
    URL.revokeObjectURL(url);
  }

  openMediaViewer(source, type) {
    // Remove existing viewer
    document.getElementById('media-viewer')?.remove();

    const viewer = document.createElement('div');
    viewer.id = 'media-viewer';
    viewer.innerHTML = `
      <div class="viewer-controls">
        <button data-action="zoom-in" title="Zoom In">+</button>
        <button data-action="zoom-out" title="Zoom Out">−</button>
        <button data-action="reset" title="Reset">⊕</button>
        <button data-action="close" title="Close">×</button>
      </div>
      <div class="viewer-content">
        <div class="viewer-transform" style="transform: scale(1) translate(0px, 0px);">
          ${type === 'image' 
            ? `<img src="${source}" draggable="false">`
            : `<div class="mermaid-clone">${source.innerHTML}</div>`
          }
        </div>
      </div>
    `;

    document.body.appendChild(viewer);

    // State
    let scale = 1;
    let translateX = 0;
    let translateY = 0;
    let isDragging = false;
    let startX, startY;

    const transform = viewer.querySelector('.viewer-transform');
    const content = viewer.querySelector('.viewer-content');

    const updateTransform = () => {
      transform.style.transform = `scale(${scale}) translate(${translateX}px, ${translateY}px)`;
    };

    // Controls
    viewer.querySelector('[data-action="zoom-in"]').addEventListener('click', () => {
      scale = Math.min(scale + 0.25, 5);
      updateTransform();
    });

    viewer.querySelector('[data-action="zoom-out"]').addEventListener('click', () => {
      scale = Math.max(scale - 0.25, 0.25);
      updateTransform();
    });

    viewer.querySelector('[data-action="reset"]').addEventListener('click', () => {
      scale = 1;
      translateX = 0;
      translateY = 0;
      updateTransform();
    });

    viewer.querySelector('[data-action="close"]').addEventListener('click', () => {
      viewer.remove();
    });

    // Mouse wheel zoom
    content.addEventListener('wheel', (e) => {
      e.preventDefault();
      const delta = e.deltaY > 0 ? -0.1 : 0.1;
      scale = Math.max(0.25, Math.min(5, scale + delta));
      updateTransform();
    });

    // Drag to pan
    content.addEventListener('mousedown', (e) => {
      isDragging = true;
      startX = e.clientX - translateX * scale;
      startY = e.clientY - translateY * scale;
      content.style.cursor = 'grabbing';
    });

    document.addEventListener('mousemove', (e) => {
      if (!isDragging) return;
      translateX = (e.clientX - startX) / scale;
      translateY = (e.clientY - startY) / scale;
      updateTransform();
    });

    document.addEventListener('mouseup', () => {
      isDragging = false;
      content.style.cursor = 'grab';
    });

    // ESC to close
    const escHandler = (e) => {
      if (e.key === 'Escape') {
        viewer.remove();
        document.removeEventListener('keydown', escHandler);
      }
    };
    document.addEventListener('keydown', escHandler);
  }

  escapeHtml(text) {
    return text
      .replace(/&/g, '&amp;')
      .replace(/</g, '&lt;')
      .replace(/>/g, '&gt;');
  }

  markdownToHtml(text) {
    console.log('📄 MarkaJiap: markdownToHtml called, input length:', text.length);
    
    // Normalize line endings to LF
    let html = text.replace(/\r\n/g, '\n').replace(/\r/g, '\n');
    console.log('📄 MarkaJiap: After line ending normalization, length:', html.length);

    // Check for mermaid in raw text (various patterns)
    const hasMermaid1 = html.includes('```mermaid');
    const hasMermaid2 = html.includes('```mermaid\n');
    const hasMermaid3 = /```mermaid/i.test(html);
    console.log('📄 MarkaJiap: Contains ```mermaid:', hasMermaid1);
    console.log('📄 MarkaJiap: Contains ```mermaid\\n:', hasMermaid2);
    console.log('📄 MarkaJiap: Regex test /```mermaid/i:', hasMermaid3);
    
    // Log first occurrence position
    const mermaidIndex = html.indexOf('```mermaid');
    if (mermaidIndex >= 0) {
      console.log('📄 MarkaJiap: ```mermaid found at index:', mermaidIndex);
      console.log('📄 MarkaJiap: Context around it:', JSON.stringify(html.substring(mermaidIndex, mermaidIndex + 100)));
    }
    
    // Extract mermaid blocks FIRST (before other code blocks)
    // Support both with and without newline after ```mermaid
    let mermaidCount = 0;
    html = html.replace(/```mermaid\s*\n([\s\S]*?)```/gi, (m, code) => {
      mermaidCount++;
      console.log('📄 MarkaJiap: Found mermaid block #' + mermaidCount);
      console.log('📄 MarkaJiap: Mermaid code:', code.substring(0, 100) + '...');
      return `<div class="mermaid">${code.trim()}</div>`;
    });
    
    console.log('📄 MarkaJiap: Total mermaid blocks extracted:', mermaidCount);
    
    // Also check if mermaid divs were created
    const mermaidDivCount = (html.match(/class="mermaid"/g) || []).length;
    console.log('📄 MarkaJiap: Mermaid divs in HTML:', mermaidDivCount);

    // Code blocks (exclude already processed mermaid)
    html = html.replace(/```(\w*)\s*\n([\s\S]*?)```/g, (m, lang, code) => {
      // Skip if this looks like it was a mermaid block
      if (lang && lang.toLowerCase() === 'mermaid') {
        console.log('📄 MarkaJiap: Skipping mermaid in code block handler');
        return m;
      }
      return `<pre data-lang="${lang || 'code'}"><code>${this.escapeHtml(code)}</code></pre>`;
    });

    // Inline code
    html = html.replace(/`([^`]+)`/g, '<code>$1</code>');

    // Headers
    html = html.replace(/^######\s+(.*)$/gm, '<h6 id="$1">$1</h6>');
    html = html.replace(/^#####\s+(.*)$/gm, '<h5 id="$1">$1</h5>');
    html = html.replace(/^####\s+(.*)$/gm, '<h4 id="$1">$1</h4>');
    html = html.replace(/^###\s+(.*)$/gm, '<h3 id="$1">$1</h3>');
    html = html.replace(/^##\s+(.*)$/gm, '<h2 id="$1">$1</h2>');
    html = html.replace(/^#\s+(.*)$/gm, '<h1 id="$1">$1</h1>');

    // Bold/Italic
    html = html.replace(/\*\*\*([^*]+)\*\*\*/g, '<strong><em>$1</em></strong>');
    html = html.replace(/\*\*([^*]+)\*\*/g, '<strong>$1</strong>');
    html = html.replace(/__([^_]+)__/g, '<strong>$1</strong>');
    html = html.replace(/\*([^*]+)\*/g, '<em>$1</em>');
    html = html.replace(/_([^_]+)_/g, '<em>$1</em>');
    html = html.replace(/~~([^~]+)~~/g, '<del>$1</del>');

    // Links & Images
    html = html.replace(/!\[([^\]]*)\]\(([^)]+)\)/g, '<img src="$2" alt="$1" loading="lazy">');
    html = html.replace(/\[([^\]]+)\]\(([^)]+)\)/g, '<a href="$2" target="_blank" rel="noopener">$1</a>');

    // Horizontal rules
    html = html.replace(/^---$/gm, '<hr>');
    html = html.replace(/^\*\*\*$/gm, '<hr>');

    // Blockquotes
    html = html.replace(/^>\s+(.*)$/gm, '<blockquote><p>$1</p></blockquote>');

    // Task lists
    html = html.replace(/^(\s*)- \[x\]\s+(.*)$/gm, '<li class="task-list-item"><input type="checkbox" checked disabled> $2</li>');
    html = html.replace(/^(\s*)- \[ \]\s+(.*)$/gm, '<li class="task-list-item"><input type="checkbox" disabled> $2</li>');

    // Lists
    html = html.replace(/^(\s*)[-*]\s+(.*)$/gm, '<li>$2</li>');
    html = html.replace(/^(\s*)\d+\.\s+(.*)$/gm, '<li>$2</li>');

    // Tables
    html = this.processTable(html);

    // Paragraphs
    html = html.split(/\n\n+/).map(block => {
      block = block.trim();
      if (!block) return '';
      if (/^<(h[1-6]|pre|blockquote|ul|ol|table|hr|li|div)/.test(block)) return block;
      if (/^<\//.test(block)) return block;
      return `<p>${block.replace(/\n/g, '<br>')}</p>`;
    }).join('\n');

    // Wrap lists
    html = html.replace(/(<li[\s\S]*?<\/li>)+/g, '<ul>$&</ul>');

    return html;
  }

  processTable(html) {
    const lines = html.split('\n');
    let inTable = false;
    let result = [];

    for (let i = 0; i < lines.length; i++) {
      const line = lines[i];
      if (/^\|(.+)\|$/.test(line)) {
        const cells = line.slice(1, -1).split('|').map(c => c.trim());
        if (cells.every(c => /^[-:]+$/.test(c))) continue;

        if (!inTable) {
          result.push('<table>');
          result.push('<tr>' + cells.map(c => `<th>${c}</th>`).join('') + '</tr>');
          inTable = true;
        } else {
          result.push('<tr>' + cells.map(c => `<td>${c}</td>`).join('') + '</tr>');
        }
      } else {
        if (inTable) {
          result.push('</table>');
          inTable = false;
        }
        result.push(line);
      }
    }

    if (inTable) result.push('</table>');
    return result.join('\n');
  }

  generateToc() {
    const toc = document.getElementById('markajiap-toc');
    const headings = document.querySelectorAll('#markajiap-content h1, #markajiap-content h2, #markajiap-content h3');

    if (headings.length === 0) {
      toc.innerHTML = '<p style="color:var(--text-secondary);font-size:13px;padding:8px;">No headings</p>';
      return;
    }

    let html = '';
    headings.forEach((h, i) => {
      const level = parseInt(h.tagName[1]);
      const text = h.textContent;
      const id = `h-${i}`;
      h.id = id;
      html += `<a href="#${id}" class="level-${level}">${text}</a>`;
    });

    toc.innerHTML = html;

    toc.querySelectorAll('a').forEach(a => {
      a.addEventListener('click', (e) => {
        e.preventDefault();
        const target = document.querySelector(a.getAttribute('href'));
        if (target) target.scrollIntoView({ behavior: 'smooth' });
      });
    });
  }

  setupInteractions(options) {
    // Export button
    document.getElementById('btn-export')?.addEventListener('click', async () => {
      // Load exporter script
      if (!window.MarkaJiapExporter) {
        const script = document.createElement('script');
        script.src = chrome.runtime.getURL('lib/exporter.js');
        document.body.appendChild(script);
        await new Promise(r => setTimeout(r, 100));
      }
      
      if (window.MarkaJiapExporter) {
        window.MarkaJiapExporter.showExportDialog();
      }
    });

    // Auto-refresh toggle
    document.getElementById('btn-refresh')?.addEventListener('click', () => {
      if (this.autoRefreshEnabled) {
        this.stopAutoRefresh();
        document.getElementById('btn-refresh').style.opacity = '0.5';
      } else {
        this.startAutoRefresh();
        document.getElementById('btn-refresh').style.opacity = '1';
      }
    });

    document.getElementById('btn-theme')?.addEventListener('click', () => {
      const current = document.documentElement.getAttribute('data-theme');
      const next = current === 'dark' ? 'light' : 'dark';
      document.documentElement.setAttribute('data-theme', next);
      chrome.storage.sync.set({ theme: next });
    });

    document.getElementById('btn-toc')?.addEventListener('click', () => {
      document.getElementById('markajiap-sidebar')?.classList.toggle('collapsed');
    });

    document.getElementById('toggle-sidebar')?.addEventListener('click', () => {
      document.getElementById('markajiap-sidebar')?.classList.toggle('collapsed');
    });

    document.getElementById('btn-print')?.addEventListener('click', () => window.print());

    const scrollBtn = document.getElementById('scroll-top');
    window.addEventListener('scroll', () => {
      if (window.scrollY > 300) {
        scrollBtn?.classList.add('visible');
      } else {
        scrollBtn?.classList.remove('visible');
      }

      const winScroll = document.documentElement.scrollTop;
      const height = document.documentElement.scrollHeight - document.documentElement.clientHeight;
      const scrolled = height > 0 ? (winScroll / height) * 100 : 0;
      const fill = document.querySelector('.progress-fill');
      if (fill) fill.style.width = scrolled + '%';
    });

    scrollBtn?.addEventListener('click', () => window.scrollTo({ top: 0, behavior: 'smooth' }));

    // Keyboard shortcuts
    document.addEventListener('keydown', (e) => {
      // Skip if typing in input/textarea
      if (e.target.tagName === 'INPUT' || e.target.tagName === 'TEXTAREA') return;
      
      // Ctrl/Cmd + B: Toggle sidebar
      if ((e.ctrlKey || e.metaKey) && e.key === 'b') {
        e.preventDefault();
        document.getElementById('markajiap-sidebar')?.classList.toggle('collapsed');
      }
      
      // Ctrl/Cmd + P: Print
      if ((e.ctrlKey || e.metaKey) && e.key === 'p') {
        e.preventDefault();
        window.print();
      }
      
      // T: Toggle theme (without modifiers)
      if (e.key === 't' && !e.ctrlKey && !e.metaKey && !e.altKey) {
        const current = document.documentElement.getAttribute('data-theme');
        const next = current === 'dark' ? 'light' : 'dark';
        document.documentElement.setAttribute('data-theme', next);
        chrome.storage.sync.set({ theme: next });
      }
      
      // E: Export dialog
      if (e.key === 'e' && !e.ctrlKey && !e.metaKey && !e.altKey) {
        document.getElementById('btn-export')?.click();
      }
      
      // R: Toggle auto-refresh
      if (e.key === 'r' && !e.ctrlKey && !e.metaKey && !e.altKey) {
        document.getElementById('btn-refresh')?.click();
      }
      
      // Escape: Close modals/lightbox
      if (e.key === 'Escape') {
        document.getElementById('markajiap-export-dialog')?.remove();
        if (window.MarkaJiapLightbox?.isOpen) {
          window.MarkaJiapLightbox.close();
        }
      }
      
      // Home: Scroll to top
      if (e.key === 'Home') {
        window.scrollTo({ top: 0, behavior: 'smooth' });
      }
      
      // End: Scroll to bottom
      if (e.key === 'End') {
        window.scrollTo({ top: document.body.scrollHeight, behavior: 'smooth' });
      }
      
      // J/K: Next/prev heading (vim-style)
      if (e.key === 'j' || e.key === 'k') {
        const headings = document.querySelectorAll('#markajiap-content h1, #markajiap-content h2, #markajiap-content h3');
        if (headings.length === 0) return;
        
        const scrollY = window.scrollY + 100;
        let targetIndex = -1;
        
        if (e.key === 'j') {
          // Find next heading below current scroll
          for (let i = 0; i < headings.length; i++) {
            if (headings[i].offsetTop > scrollY) {
              targetIndex = i;
              break;
            }
          }
        } else {
          // Find previous heading above current scroll
          for (let i = headings.length - 1; i >= 0; i--) {
            if (headings[i].offsetTop < scrollY - 50) {
              targetIndex = i;
              break;
            }
          }
        }
        
        if (targetIndex >= 0 && targetIndex < headings.length) {
          headings[targetIndex].scrollIntoView({ behavior: 'smooth', block: 'start' });
        }
      }
    });

    // Store initial content hash and start auto-refresh
    this.contentHash = this.simpleHash(this.currentMarkdown || '');
    this.startAutoRefresh();
  }

  restoreOriginal() {
    if (!this.isRendered) return;

    document.getElementById('markajiap-styles')?.remove();
    document.body.innerHTML = this.originalContent;
    document.title = this.originalTitle;
    document.documentElement.removeAttribute('data-theme');

    this.isRendered = false;

    setTimeout(() => {
      this.showFloatingButton();
      this.updateButton(false);
    }, 100);
  }
}

// Initialize
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', () => {
    window.markaJiapContent = new MarkaJiapContent();
  });
} else {
  window.markaJiapContent = new MarkaJiapContent();
}
