// MarkaJiap - Background Service Worker

// Initialize extension
chrome.runtime.onInstalled.addListener((details) => {
  if (details.reason === 'install') {
    // Set default options
    chrome.storage.sync.set({
      theme: 'light',
      autoRender: true,
      showToc: true,
      codeLineNumbers: true,
      enableMermaid: true,
      enableMath: true,
      fontSize: 16,
      fontFamily: 'IBM Plex Sans',
      customCss: ''
    });

    // Open welcome page
    chrome.tabs.create({
      url: 'options.html#welcome'
    });
  }
  
  // Create context menus
  createContextMenus();
});

// Create context menus
function createContextMenus() {
  chrome.contextMenus.removeAll(() => {
    // View selection as Markdown
    chrome.contextMenus.create({
      id: 'viewSelection',
      title: 'Render Selection as Markdown',
      contexts: ['selection']
    });

    // View link as Markdown
    chrome.contextMenus.create({
      id: 'viewLink',
      title: 'Open Link with MarkaJiap',
      contexts: ['link'],
      targetUrlPatterns: [
        '*://*/*.md',
        '*://*/*.markdown',
        '*://raw.githubusercontent.com/*',
        '*://gist.githubusercontent.com/*'
      ]
    });

    // View page as Markdown
    chrome.contextMenus.create({
      id: 'viewPage',
      title: 'View Page with MarkaJiap',
      contexts: ['page']
    });
  });
}

// Handle context menu clicks
chrome.contextMenus.onClicked.addListener((info, tab) => {
  switch (info.menuItemId) {
    case 'viewSelection':
      openViewer({ type: 'text', content: info.selectionText });
      break;
    
    case 'viewLink':
      openViewer({ type: 'url', url: info.linkUrl });
      break;
    
    case 'viewPage':
      chrome.tabs.sendMessage(tab.id, { action: 'getPageMarkdown' }, (response) => {
        if (response?.markdown) {
          openViewer({ type: 'text', content: response.markdown });
        }
      });
      break;
  }
});

// Handle messages from popup and content scripts
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  switch (request.action) {
    case 'openViewer':
      openViewer(request.data);
      sendResponse({ success: true });
      break;
    
    case 'fetchUrl':
      fetchMarkdown(request.url)
        .then(content => sendResponse({ success: true, content }))
        .catch(error => sendResponse({ success: false, error: error.message }));
      return true; // Keep channel open for async response
    
    case 'getOptions':
      chrome.storage.sync.get(null, (options) => {
        sendResponse({ success: true, options });
      });
      return true;
    
    case 'setOptions':
      chrome.storage.sync.set(request.options, () => {
        sendResponse({ success: true });
      });
      return true;
    
    case 'exportPdf':
      exportToPdf(request.html, request.filename);
      sendResponse({ success: true });
      break;
    
    case 'exportHtml':
      exportToHtml(request.html, request.filename);
      sendResponse({ success: true });
      break;
  }
});

// Open viewer in new tab with data
async function openViewer(data) {
  // Store data temporarily
  await chrome.storage.local.set({ viewerData: data });
  
  // Create viewer tab
  chrome.tabs.create({
    url: chrome.runtime.getURL('viewer.html')
  });
}

// Fetch markdown from URL
async function fetchMarkdown(url) {
  const response = await fetch(url, {
    headers: {
      'Accept': 'text/plain, text/markdown, */*'
    }
  });
  
  if (!response.ok) {
    throw new Error(`Failed to fetch: ${response.status} ${response.statusText}`);
  }
  
  return await response.text();
}

// Export to PDF using Chrome's print API
function exportToPdf(html, filename) {
  // Create a temporary tab with the HTML
  const blob = new Blob([html], { type: 'text/html' });
  const url = URL.createObjectURL(blob);
  
  chrome.tabs.create({ url, active: true }, (tab) => {
    // Wait for the tab to load, then print
    chrome.tabs.onUpdated.addListener(function listener(tabId, changeInfo) {
      if (tabId === tab.id && changeInfo.status === 'complete') {
        chrome.tabs.onUpdated.removeListener(listener);
        
        // Open print dialog
        chrome.scripting.executeScript({
          target: { tabId: tab.id },
          func: () => window.print()
        });
        
        // Clean up blob URL
        URL.revokeObjectURL(url);
      }
    });
  });
}

// Export to HTML file
function exportToHtml(html, filename) {
  const blob = new Blob([html], { type: 'text/html' });
  const url = URL.createObjectURL(blob);
  
  chrome.downloads.download({
    url: url,
    filename: filename || 'markajiap-export.html',
    saveAs: true
  }, () => {
    URL.revokeObjectURL(url);
  });
}

// Handle keyboard shortcuts (only if commands API available)
if (chrome.commands && chrome.commands.onCommand) {
  chrome.commands.onCommand.addListener((command) => {
    switch (command) {
      case 'toggle-viewer':
        chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
          chrome.tabs.sendMessage(tabs[0].id, { action: 'toggleViewer' });
        });
        break;
    }
  });
}

// Listen for tab updates to auto-detect markdown files
chrome.tabs.onUpdated.addListener((tabId, changeInfo, tab) => {
  if (changeInfo.status === 'complete' && tab.url) {
    const url = tab.url.toLowerCase();
    
    // Check if it's a markdown file or raw content
    if (url.endsWith('.md') || 
        url.endsWith('.markdown') ||
        url.includes('raw.githubusercontent.com') ||
        url.includes('gist.githubusercontent.com')) {
      
      // Get auto-render setting
      chrome.storage.sync.get(['autoRender'], (result) => {
        if (result.autoRender !== false) {
          chrome.tabs.sendMessage(tabId, { action: 'autoRender' });
        }
      });
    }
  }
});

// Badge for showing status
function updateBadge(text, color = '#7B2D8E') {
  chrome.action.setBadgeText({ text });
  chrome.action.setBadgeBackgroundColor({ color });
}

// Clear badge after 3 seconds
function flashBadge(text, color = '#7B2D8E') {
  updateBadge(text, color);
  setTimeout(() => updateBadge(''), 3000);
}
